from datetime import datetime
# Werkzeug库的security实现散列密码的计算
from werkzeug.security import generate_password_hash, check_password_hash
from itsdangerous import TimedJSONWebSignatureSerializer as Serializer
from apps import constants
from . import db
from config import Config
from sqlalchemy import text


class BaseModel:
    """
    模型基类，补充创建时间与更新时间
    """
    create_time = db.Column(db.DateTime, default=datetime.now, doc='创建时间', comment='创建时间')  # 记录的创建时间
    update_time = db.Column(db.DateTime, default=datetime.now, onupdate=datetime.now, doc='更新时间',
                            comment='更新时间')  # 记录的更新时间


# 招商驾驶舱开工项目产业分布数据
class IndustryDistribute(db.Model):
    __tablename_ = "industry_distribute"
    __table_args__ = ({'comment': '招商驾驶舱开工项目产业分布数据'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    industry_name = db.Column(db.String(20), comment='行业名称')
    industry_num = db.Column(db.Integer, comment='个数')
    investment_volume = db.Column(db.Float, comment='投资额（亿元）')


# 新版产业链数据
class IndustryChain(db.Model):
    __tablename_ = "industry_chain"
    __table_args__ = ({'comment': '新版产业链数据数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    industry_name = db.Column(db.String(20), comment='行业名称')
    icon_url = db.Column(db.String(255), comment='未选中时行业图标')
    icon_url1 = db.Column(db.String(255), comment='选中时行业图标')
    status = db.Column(db.Integer, doc='启用状态1启用，2禁用', comment='启用状态1启用，2禁用')  # 启用状态1启用，2禁用
    industry_type = db.Column(db.Integer, comment='行业类型(0是产业集群，1是上游行业，2是中游行业，3是下游行业)')
    relate_id = db.Column(db.Integer, comment='关系id(关联哪个产业环节')
    chain_id = db.Column(db.Integer, comment='关系id(关联哪个产业链')
    enterprise = db.relationship('Enterprise', backref=db.backref('enterprise'))
    enterprise_num = db.Column(db.Integer, comment='相关全企业表数量')


# 全国企业
class Enterprise(db.Model):
    __tablename__ = "enterprise"

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='企业主键id', comment='企业主键id主键id')  # 企业
    industry_chain_id = db.Column(db.Integer, db.ForeignKey('industry_chain.id'))
    company_name = db.Column(db.String(255), index=True, doc='企业名', comment='企业名')  # 企业名
    status = db.Column(db.String(32), doc='经营状态', comment='经营状态')  # 经营状态
    legal = db.Column(db.String(255), doc='发定代表人', comment='发定代表人')  # 发定代表人
    capital = db.Column(db.String(255), doc='注册资本,22万美元', comment='注册资本,22万美元')  # 注册资本,22万美元
    capital_nums = db.Column(db.Float, doc='注册资本转换成人民币数值', comment='注册资本转换成人民币数值')  # 注册资本转换成人民币数值
    capital_id = db.Column(db.Integer, doc='注册资本大小类型，0-100,1,100-500,2,500-1000,3,1000-5000,4,5000-10000,5,10000+，6',
                           comment='注册资本大小类型，0-100,1,100-500,2,500-1000,3,1000-5000,4,5000-10000,5,10000+，6')  # 注册资本大小类型，0-100,1,100-500,2,500-1000,3,1000-5000,4,5000-10000,5,10000+，6
    build_date = db.Column(db.DateTime, doc='注册时间', comment='注册时间')  # 注册时间
    yearid = db.Column(db.Integer, index=True, doc='成立时间年限id(1-3，1)(3-5，2)(5-8，3)（8-10，4）（10-15,5）（15以上，6）',
                       comment='成立时间年限id(1-3，1)(3-5，2)(5-8，3)（8-10，4）（10-15,5）（15以上，6）')  # 成立时间年限id(1-3，1)(3-5，2)(5-8，3)（8-10，4）（10-15,5）（15以上，6）
    province = db.Column(db.String(32), index=True, doc='省', comment='省')  # 省
    city = db.Column(db.String(32), index=True, doc='市', comment='市')  # 市
    district = db.Column(db.String(32), index=True, doc='区', comment='区')  # 区
    lng = db.Column(db.String(100), doc='经度', comment='经度')  # 经度
    lat = db.Column(db.String(100), doc='纬度', comment='纬度')  # 纬度
    p_lng = db.Column(db.String(100), doc='省经度', comment='省经度')  # 省经度
    p_lat = db.Column(db.String(100), doc='省纬度', comment='省纬度')  # 省纬度
    c_lng = db.Column(db.String(100), doc='市经度', comment='市经度')  # 市经度
    c_lat = db.Column(db.String(100), doc='市纬度', comment='市纬度')  # 市纬度
    d_lng = db.Column(db.String(100), doc='区经度', comment='区经度')  # 区经度
    d_lat = db.Column(db.String(100), doc='区纬度', comment='区纬度')  # 区纬度
    address = db.Column(db.String(255), doc='企业地址', comment='企业地址')  # 企业地址
    telephone = db.Column(db.Text, doc='电话', comment='电话')  # 电话
    telephone_more = db.Column(db.Text, doc='更多电话', comment='更多电话')  # 更多电话
    email = db.Column(db.Text, doc='邮箱', comment='邮箱')  # 邮箱
    social_code = db.Column(db.String(100), doc='统一社会信用代码', comment='统一社会信用代码')  # 统一社会信用代码
    tax_code = db.Column(db.String(100), doc='纳税人识别号', comment='纳税人识别号')  # 纳税人识别号
    register_code = db.Column(db.String(100), doc='注册号', comment='注册号')  # 注册号
    company_code = db.Column(db.String(100), doc='组织机构代码', comment='组织机构代码')  # 组织机构代码
    bao_num = db.Column(db.Integer, doc='参保人数', comment='参保人数')  # 参保人数
    entype = db.Column(db.String(100), doc='企业类型', comment='企业类型')  # 企业类型
    entypeid = db.Column(db.Integer, index=True,
                         doc='公司类型id(个人独资企业-1)(股份有限公司-2)(有限责任公司-3)(合伙企业-4)(集体所有制-5)(全民所有制企业-6)(外商企业-7)',
                         comment='公司类型id(个人独资企业-1)(股份有限公司-2)(有限责任公司-3)(合伙企业-4)(集体所有制-5)(全民所有制企业-6)(外商企业-7)')  # 公司类型id(个人独资企业-1)(股份有限公司-2)(有限责任公司-3)(合伙企业-4)(集体所有制-5)(全民所有制企业-6)(外商企业-7)
    scale_range = db.Column(db.Integer, index=True,
                            doc='企业规模id，1：20人以下，2:20-99人，3:100-499人，4：500-999人，5： 1000-4999人，6：5000-9999人，7:10000人',
                            comment='企业规模id，1：20人以下，2:20-99人，3:100-499人，4：500-999人，5： 1000-4999人，6：5000-9999人，7:10000人')

    company_industry = db.Column(db.String(100), doc='所属行业', comment='所属行业')  # 所属行业
    web_site = db.Column(db.String(255), doc='企业网址', comment='企业网址')  # 企业网址
    business_scope = db.Column(db.Text, doc='企业经营范围', comment='企业经营范围')  # 企业经营范围
    money_type = db.Column(db.String(100), doc='注册币种', comment='注册币种')  # 注册币种
    money_type_id = db.Column(db.Integer, doc='注册币种类型id', comment='注册币种类型id')  # 注册币种类型id

    high_new = db.Column(db.String(32), doc='是否高新技术企业', comment='是否高新技术企业')  # 是否高新技术企业
    parti_year = db.Column(db.Integer, doc='高新企业注册年份', comment='高新企业注册年份')  # 高新企业注册年份
    tbe = db.Column(db.String(32), doc='是否科技型中小企业', comment='是否科技型中小企业')  # 是否科技型中小企业
    tbe_sjmy = db.Column(db.String(32), doc='是否为省级民营科技企业', comment='是否为省级民营科技企业')  # 省级民营科技企业
    zjtg = db.Column(db.String(32), doc='是否为专精特新企业', comment='是否为专精特新企业')  # 山西专精特工企业
    zjtg_gjjxjr = db.Column(db.String(32), doc='是否为国家级专精特新小巨人企业', comment='是否为国家级专精特新小巨人企业')  # 国家级专精特新小巨人企业
    zjtg_sjxjr = db.Column(db.String(32), doc='是否为省级专精特新小巨人企业', comment='是否为省级专精特新小巨人企业')  # 省级专精特新小巨人企业
    fianacing = db.Column(db.String(32), doc='是否为有融资企业', comment='是否为有融资企业')  # 是否为有融资企业
    fianacing_rounds = db.Column(db.String(32), doc='融资轮次', comment='融资轮次')  # 融资轮次
    roundid = db.Column(db.Integer, index=True,
                        doc='融资轮次id(天使/种子，1)（PreA/A+,2）（PreB/B+,3）(C轮以上，4)（收并购，5）（战略投资，6）（其他，7）',
                        comment='融资轮次id(天使/种子，1)（PreA/A+,2）（PreB/B+,3）(C轮以上，4)（收并购，5）（战略投资，6）（其他，7）')  # 融资轮次id(天使/种子，1)（PreA/A+,2）（PreB/B+,3）(C轮以上，4)（收并购，5）（战略投资，6）（其他，7）
    financing_amount = db.Column(db.Float, doc='融资金额', comment='融资金额')  # 融资金额
    software_copyright = db.Column(db.String(32), doc='是否为有软件著作权', comment='是否为有软件著作权')  # 是否为有软件著作权
    num_software = db.Column(db.Integer, doc='软件著作权个数', comment='软件著作权个数')  # 软件著作权个数
    quoted_company = db.Column(db.String(32), doc='是否上市企业', comment='是否上市企业')  # 是否上市企业
    public_sector = db.Column(db.String(32), doc='上市板块', comment='上市板块')  # 上市板块
    public_id = db.Column(db.Integer, index=True, doc='上市版块的id----360企业画像（A股,1）(创业股，2)(港股，3)(新三股，4)(新四股,5)(中小板,6)',
                          comment='上市版块的id----360企业画像（A股,1）(创业股，2)(港股，3)(新三股，4)(新四股,5)(中小板,6)')  # 上市版块的id----360企业画像（A股,1）(创业股，2)(港股，3)(新三股，4)(新四股,5)(中小板,6)

    foreign_investment = db.Column(db.String(32), doc='是否外商投资', comment='是否外商投资')  # 是否外商投资
    patent = db.Column(db.String(32), doc='是否为有专利企业', comment='是否为有专利企业')  # 是否为有专利企业
    num_patent = db.Column(db.Integer, doc='专利个数', comment='专利个数')  # 专利个数
    company_info = db.Column(db.Text, doc='公司简介', comment='公司简介')  # 公司简介

    dengl = db.Column(db.String(32), doc='瞪羚', comment='瞪羚')  # 瞪羚
    unicorn = db.Column(db.String(32), doc='独角兽企业', comment='独角兽企业')  # 独角兽企业
    isfive = db.Column(db.String(32), doc='是否中国500强', comment='是否中国500强')  # 是否中国500强

    takingn = db.Column(db.Float, doc='营收', comment='营收')  # 营收

    hots = db.Column(db.Integer, doc='企业热度(权重值)', comment='企业热度(权重值)')  # 企业热度(权重值)
    c_name = db.Column(db.String(255), doc='行业类名', comment='行业类名')  # 行业类名
    c_type = db.Column(db.Integer, index=True, doc='行业类id', comment='行业类id')  # 行业类id
    product_all = db.Column(db.String(255), doc='公司拥有产品', comment='公司拥有产品')  # 公司拥有产品

    c_name1 = db.Column(db.String(255), doc='行业类名', comment='行业类名')  # 行业类名
    c_type1 = db.Column(db.Integer, index=True, doc='行业类id', comment='行业类id')  # 行业类id
    product_all1 = db.Column(db.String(255), doc='公司拥有产品', comment='公司拥有产品')  # 公司拥有产品

    c_name2 = db.Column(db.String(255), doc='行业类名', comment='行业类名')  # 行业类名
    c_type2 = db.Column(db.Integer, index=True, doc='行业类id', comment='行业类id')  # 行业类id
    product_all2 = db.Column(db.String(255), doc='公司拥有产品', comment='公司拥有产品')  # 公司拥有产品

    f_name = db.Column(db.String(255), doc='父行业类名', comment='父行业类名')  # 父行业类名
    f_type = db.Column(db.Integer, index=True, doc='父行业类id', comment='父行业类id')  # 父行业类id

    sxonhun = db.Column(db.String(32), comment='是否山西100强')  # 是否山西100强
    scale = db.Column(db.String(32), comment='规模以上企业')  # 规模以上企业
    serve = db.Column(db.String(32), comment='限额以上服务业')  # 限额以上服务业

    __table_args__ = ({'comment': '全国企业表'})  # 添加表注释


# 晋城企业表
class Company(db.Model):
    __tablename__ = "company"
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='企业主键id', comment='企业主键id')  # 企业
    company_name = db.Column(db.String(255), doc='企业名', comment='企业名')  # 企业名
    status = db.Column(db.String(32), doc='经营状态', comment='经营状态')  # 经营状态
    legal = db.Column(db.String(32), doc='法定代表人', comment='法定代表人')  # 法定代表人
    capital = db.Column(db.String(255), doc='注册资本,22万美元', comment='注册资本,22万美元')  # 注册资本,22万美元
    capital_nums = db.Column(db.Float, doc='注册资本转换成人民币数值', comment='注册资本转换成人民币数值')  # 注册资本转换成人民币数值
    capital_id = db.Column(db.Integer, doc='注册资本大小类型，0-100,1,100-500,2,500-1000,3,1000-5000,4,5000-10000,5,10000+，6',
                           comment='注册资本大小类型，0-100,1,100-500,2,500-1000,3,1000-5000,4,5000-10000,5,10000+，6')  # 注册资本大小类型，0-100,1,100-500,2,500-1000,3,1000-5000,4,5000-10000,5,10000+，6
    build_date = db.Column(db.DateTime, doc='注册时间', comment='注册时间')  # 注册时间
    yearid = db.Column(db.Integer, doc='成立时间年限id(1-3，1)(3-5，2)(5-8，3)（8-10，4）（10-15,5）（15以上，6）',
                       comment='成立时间年限id(1-3，1)(3-5，2)(5-8，3)（8-10，4）（10-15,5）（15以上，6）')  # 成立时间年限id(1-3，1)(3-5，2)(5-8，3)（8-10，4）（10-15,5）（15以上，6）
    province = db.Column(db.String(32), doc='省', comment='省')  # 省
    city = db.Column(db.String(32), doc='市', comment='市')  # 市
    district = db.Column(db.String(32), doc='区', comment='区')  # 区
    lng = db.Column(db.String(100), doc='经度', comment='经度')  # 经度
    lat = db.Column(db.String(100), doc='纬度', comment='纬度')  # 纬度
    c_lng = db.Column(db.String(100), doc='市经度', comment='市经度')  # 市经度
    c_lat = db.Column(db.String(100), doc='市纬度', comment='市纬度')  # 市纬度
    d_lng = db.Column(db.String(100), doc='区经度', comment='区经度')  # 区经度
    d_lat = db.Column(db.String(100), doc='区纬度', comment='区纬度')  # 区纬度
    address = db.Column(db.String(255), doc='企业地址', comment='企业地址')  # 企业地址
    telephone = db.Column(db.Text, doc='电话', comment='电话')  # 电话
    telephone_more = db.Column(db.Text, doc='更多电话', comment='更多电话')  # 更多电话
    email = db.Column(db.Text, doc='邮箱', comment='邮箱')  # 邮箱
    social_code = db.Column(db.String(100), doc='统一社会信用代码', comment='统一社会信用代码')  # 统一社会信用代码
    tax_code = db.Column(db.String(100), doc='纳税人识别号', comment='纳税人识别号')  # 纳税人识别号
    register_code = db.Column(db.String(100), doc='注册号', comment='注册号')  # 注册号
    company_code = db.Column(db.String(100), doc='组织机构代码', comment='组织机构代码')  # 组织机构代码
    bao_num = db.Column(db.Integer, doc='参保人数', comment='参保人数')  # 参保人数
    entype = db.Column(db.String(100), doc='企业类型', comment='企业类型')  # 企业类型
    entypeid = db.Column(db.Integer, doc='公司类型id(个人独资企业-1)(股份有限公司-2)(有限责任公司-3)(合伙企业-4)(集体所有制-5)(全民所有制企业-6)(外商企业-7)',
                         comment='公司类型id(个人独资企业-1)(股份有限公司-2)(有限责任公司-3)(合伙企业-4)(集体所有制-5)(全民所有制企业-6)(外商企业-7)')  # 公司类型id(个人独资企业-1)(股份有限公司-2)(有限责任公司-3)(合伙企业-4)(集体所有制-5)
    # (全民所有制企业-6)(外商企业-7)

    company_industry = db.Column(db.String(100), doc='所属行业', comment='所属行业')  # 所属行业
    web_site = db.Column(db.String(255), doc='企业网址', comment='企业网址')  # 企业网址
    business_scope = db.Column(db.Text, doc='企业经营范围', comment='企业经营范围')  # 企业经营范围
    money_type = db.Column(db.String(100), doc='注册币种', comment='注册币种')  # 注册币种
    money_type_id = db.Column(db.Integer, doc='注册币种类型id', comment='注册币种类型id')  # 注册币种类型id

    high_new = db.Column(db.String(32), doc='是否高新技术企业', comment='是否高新技术企业')  # 是否高新技术企业
    parti_year = db.Column(db.Integer, doc='高新企业注册年份', comment='高新企业注册年份')  # 高新企业注册年份
    tbe = db.Column(db.String(32), doc='是否科技型中小企业', comment='是否科技型中小企业')  # 是否科技型中小企业
    tbe_sjmy = db.Column(db.String(32), doc='是否为省级民营科技企业', comment='是否为省级民营科技企业')  # 省级民营科技企业
    fianacing = db.Column(db.String(32), doc='是否为有融资企业', comment='是否为有融资企业')  # 是否为有融资企业
    fianacing_rounds = db.Column(db.String(32), doc='融资轮次', comment='融资轮次')  # 融资轮次
    roundid = db.Column(db.Integer, doc='融资轮次id(天使/种子，1)（PreA/A+,2）（PreB/B+,3）(C轮以上，4)（收并购，5）（战略投资，6）（其他，7）',
                        comment='融资轮次id(天使/种子，1)（PreA/A+,2）（PreB/B+,3）(C轮以上，4)（收并购，5）（战略投资，6）（其他，7）')  # 融资轮次id(天使/种子，1)（PreA/A+,2）（PreB/B+,3）(C轮以上，4)（收并购，5）（战略投资，6）（其他，7）
    financing_amount = db.Column(db.Float, doc='融资金额', comment='融资金额')  # 融资金额
    software_copyright = db.Column(db.String(32), doc='是否为有软件著作权', comment='是否为有软件著作权')  # 是否为有软件著作权
    num_software = db.Column(db.Integer, doc='软件著作权个数', comment='软件著作权个数')  # 软件著作权个数
    public_sector = db.Column(db.String(32), doc='上市板块', comment='上市板块')  # 上市板块
    quoted_company = db.Column(db.String(32), doc='是否上市企业', comment='是否上市企业')  # 是否上市企业
    public_id = db.Column(db.Integer, doc='上市版块的id----360企业画像（A股,1）(创业股，2)(港股，3)(新三股，4)(新四股,5)(中小板,6)',
                          comment='上市版块的id----360企业画像（A股,1）(创业股，2)(港股，3)(新三股，4)(新四股,5)(中小板,6)')  # 上市版块的id----360企业画像（A股,1）(创业股，2)(港股，3)(新三股，4)(新四股,5)(中小板,6)

    foreign_investment = db.Column(db.String(32), doc='是否外商投资', comment='是否外商投资')  # 是否外商投资
    patent = db.Column(db.String(32), doc='是否为有专利企业', comment='是否为有专利企业')  # 是否为有专利企业
    num_patent = db.Column(db.Integer, doc='专利个数', comment='专利个数')  # 专利个数
    induzone = db.Column(db.String(255), doc='所在园区', comment='所在园区')  # 所在园区
    company_info = db.Column(db.Text, doc='公司简介', comment='公司简介')  # 公司简介

    sxmon = db.Column(db.String(32), doc='是否山西民营100强', comment='是否山西民营100强')  # 是否山西民营100强（2019年）
    zjtg = db.Column(db.String(32), doc='是否为山西专精特新企业', comment='是否为山西专精特新企业')  # 山西专精特工企业
    zjtg_gjjxjr = db.Column(db.String(32), doc='是否为国家级专精特新小巨人企业', comment='是否为国家级专精特新小巨人企业')  # 国家级专精特新小巨人企业
    zjtg_sjxjr = db.Column(db.String(32), doc='是否为省级专精特新小巨人企业', comment='是否为省级专精特新小巨人企业')  # 省级专精特新小巨人企业
    unicorn = db.Column(db.String(32), doc='独角兽企业', comment='独角兽企业')  # 独角兽企业
    dengl = db.Column(db.String(32), doc='瞪羚', comment='瞪羚')  # 瞪羚
    hncode = db.Column(db.String(32), doc='高企证书编号', comment='高企证书编号')  # 高企证书编号
    isfive = db.Column(db.String(32), doc='是否中国500强', comment='是否中国500强')  # 是否中国500强
    istyfive = db.Column(db.String(32), doc='是否2020中国500强', comment='是否2020中国500强')  # 是否2020中国500强
    quotedate = db.Column(db.DateTime, doc='上市时间', comment='上市时间')  # 上市时间
    sxonhun = db.Column(db.String(32), doc='是否山西100强', comment='是否山西100强')  # 是否山西100强

    product_all = db.Column(db.String(255), doc='公司拥有产品', comment='公司拥有产品')  # 公司拥有产品
    scale = db.Column(db.String(32), doc='规模以上企业', comment='规模以上企业')  # 规模以上企业
    serve = db.Column(db.String(32), doc='限额以上服务业', comment='限额以上服务业')  # 限额以上服务业

    takingn = db.Column(db.Float, doc='营收 (万元)', comment='营收 (万元)')  # 2019营收 (万元)
    ouputn = db.Column(db.Float, doc='年产值', comment='年产值')  # 2019年产值
    profit = db.Column(db.Float, doc='公司利润（万元）', comment='公司利润（万元）')  # 2019公司利润（万元）
    c_name = db.Column(db.String(255), doc='行业类名', comment='行业类名')  # 行业类名
    c_type = db.Column(db.Integer, doc='行业类id', comment='行业类id')  # 行业类id
    f_name = db.Column(db.String(255), doc='父行业类名', comment='父行业类名')  # 父行业类名
    f_type = db.Column(db.Integer, doc='父行业类id', comment='父行业类id')  # 父行业类id
    hots = db.Column(db.Integer, doc='企业热度', comment='企业热度')  # 企业热度
    sort_num = db.Column(db.Integer, doc='权重值', comment='权重值')  # 权重值
    stream = db.Column(db.String(32), doc='行业上下游', comment='行业上下游')  # 行业上下游
    product = db.Column(db.String(255), doc='生产产品', comment='生产产品')  # 生产产品
    __table_args__ = ({'comment': '晋城企业表'})  # 添加表注释


# 晋城企业行政许可
class CompanyAdminPermission(db.Model):
    __tablename_ = "company_admin_permission"
    __table_args__ = ({'comment': '晋城企业行政许可数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='晋城企业行政许可主键id', comment='晋城企业行政许可主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')
    number = db.Column(db.String(32), doc='晋城企业行政许可编号', comment='晋城企业行政许可编号')
    name = db.Column(db.String(32), doc='晋城企业行政许可证名称', comment='晋城企业行政许可证名称')
    time = db.Column(db.DateTime, doc='有效期自', comment='有效期自')
    effective_data = db.Column(db.DateTime, doc='有效期至', comment='有效期至')
    Licensing_authority = db.Column(db.String(32), doc='许可机关', comment='许可机关')
    content = db.Column(db.String(32), doc='许可内容', comment='许可内容')
    source = db.Column(db.String(32), doc='来源', comment='来源')


# 晋城企业税务信用
class CompanyTaxCredit(db.Model):
    __tablename_ = "company_tax_credit"
    __table_args__ = ({'comment': '晋城企业税务信用数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='晋城企业税务信用主键id', comment='晋城企业税务信用主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')
    evaluation_annual = db.Column(db.String(32), doc='评价年度', comment='评价年度')
    identify_number = db.Column(db.String(32), doc='纳税人识别号', comment='纳税人识别号')
    level = db.Column(db.String(32), doc='纳税信用等级', comment='纳税信用等级')
    evaluate = db.Column(db.String(32), doc='单位评价', comment='单位评价')


# 晋城企业进出口信用
class CompanyImportExport(db.Model):
    __tablename_ = "company_import_export"
    __table_args__ = ({'comment': '晋城企业进出口信用数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='晋城企业进出口信用主键id', comment='晋城企业进出口信用主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')
    register_customs = db.Column(db.String(32), doc='注册海关', comment='注册海关')
    business_category = db.Column(db.String(32), doc='经营类别', comment='经营类别')
    register_date = db.Column(db.DateTime, doc='注册日期', comment='注册日期')


# 晋城企业供应商
class CompanySupplier(db.Model):
    __tablename_ = "company_supplier"
    __table_args__ = ({'comment': '晋城企业供应商数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='晋城企业供应商主键id', comment='晋城企业供应商主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')
    supplier_name = db.Column(db.String(32), doc='客户名字', comment='客户名字')
    buy_rate = db.Column(db.String(32), doc='采购占比', comment='采购占比')
    buy_money = db.Column(db.String(32), doc='采购金额', comment='采购金额')
    open_time = db.Column(db.DateTime, doc='公开时间', comment='公开时间')
    data_source = db.Column(db.String(32), doc='数据来源', comment='数据来源')
    relation = db.Column(db.String(32), doc='关联关系', comment='关联关系')
    lng = db.Column(db.String(100), doc='经度', comment='经度')  # 经度
    lat = db.Column(db.String(100), doc='纬度', comment='纬度')  # 纬度


# 晋城企业客户
class CompanyCustomer(db.Model):
    __tablename_ = "company_customer"
    __table_args__ = ({'comment': '晋城企业客户数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='晋城企业客户主键id', comment='晋城企业客户主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')
    customer_name = db.Column(db.String(32), doc='客户', comment='客户')
    sales_rate = db.Column(db.String(32), doc='销售占比', comment='销售占比')
    sales_money = db.Column(db.String(32), doc='销售金额', comment='销售金额')
    open_time = db.Column(db.DateTime, comment='公开时间')
    data_source = db.Column(db.String(32), doc='数据来源', comment='数据来源')
    relation = db.Column(db.String(32), doc='关联关系', comment='关联关系')


# 晋城企业详情股权出质
class CompanyStock(db.Model):
    __tablename_ = "company_stock"
    __table_args__ = ({'comment': '晋城企业详情股权出质'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='股权出质主键id', comment='股权出质主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')

    num = db.Column(db.String(32), doc='等级编号', comment='等级编号')
    person = db.Column(db.String(10), doc='出质人', comment='出质人')
    from_company = db.Column(db.String(25), doc='出质股权的企业', comment='出质股权的企业')
    pledgee = db.Column(db.String(10), doc='质权人', comment='质权人')
    amount = db.Column(db.Float, doc='出质股权数额（万元）', comment='出质股权数额（万元）')
    datatime = db.Column(db.DateTime, doc='登记日期', comment='登记日期')
    status = db.Column(db.String(10), doc='状态', comment='状态')


# 晋城企业详情股权质押
class CompanyPledge(db.Model):
    __tablename_ = "company_pledge"
    __table_args__ = ({'comment': '晋城企业详情股权质押'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='股权质押主键id', comment='股权质押主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')

    person1 = db.Column(db.String(10), doc='质押人', comment='质押人')
    jion_company = db.Column(db.String(25), doc='参股企业', comment='参股企业')
    person2 = db.Column(db.String(10), doc='质押权人', comment='质押权人')
    number = db.Column(db.Float, doc='质押股份总额（股）', comment='质押股份总额（股）')
    amount = db.Column(db.Float, doc='质押股份市值（元）', comment='质押股份市值（元）')
    status = db.Column(db.String(10), doc='状态', comment='状态')
    datatime = db.Column(db.DateTime, doc='公告日期', comment='公告日期')


# 晋城企业详情担保信息
class CompanyDanbao(db.Model):
    __tablename_ = "company_danbao"
    __table_args__ = ({'comment': '晋城企业详情担保信息'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='担保信息主键id', comment='担保信息主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')

    person1 = db.Column(db.String(10), doc='被担保方', comment='被担保方')
    person2 = db.Column(db.String(10), doc='担保方', comment='担保方')
    method = db.Column(db.String(10), doc='方式', comment='方式')
    amount = db.Column(db.Float, doc='担保金额（万元）', comment='担保金额（万元）')
    datatime = db.Column(db.DateTime, doc='公告日期', comment='公告日期')


# 晋城企业详情专利信息
class CompanyPatent(db.Model):
    __tablename_ = "company_patent"
    __table_args__ = ({'comment': '晋城企业详情专利信息'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='专利信息主键id', comment='专利信息主键id')  # 企业
    company_id = db.Column(db.Integer, doc='晋城企业id', comment='晋城企业id')

    name = db.Column(db.String(50), doc='发明名称', comment='发明名称')
    type = db.Column(db.String(20), doc='类型', comment='类型')
    status = db.Column(db.String(20), doc='法律状态', comment='法律状态')
    num = db.Column(db.String(50), doc='申请号', comment='申请号')
    day = db.Column(db.DateTime, doc='申请日', comment='申请日')
    open_num = db.Column(db.String(20), doc='公开号', comment='公开号')
    datatime = db.Column(db.DateTime, doc='公开日期', comment='公开日期')
    person = db.Column(db.String(20), doc='发明人', comment='发明人')


# 全国企业表企业行政许可
class EnterpriseAdminPermission(db.Model):
    __tablename_ = "company_admin_permission"
    __table_args__ = ({'comment': '全国企业表企业行政许可数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='全国企业表企业行政许可主键id',
                   comment='全国企业表企业行政许可主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')
    number = db.Column(db.String(32), doc='全国企业表企业行政许可编号', comment='全国企业表企业行政许可编号')
    name = db.Column(db.String(32), doc='全国企业表企业行政许可证名称', comment='全国企业表企业行政许可证名称')
    time = db.Column(db.DateTime, doc='有效期自', comment='有效期自')
    effective_data = db.Column(db.DateTime, doc='有效期至', comment='有效期至')
    Licensing_authority = db.Column(db.String(32), doc='许可机关', comment='许可机关')
    content = db.Column(db.String(32), doc='许可内容', comment='许可内容')
    source = db.Column(db.String(32), doc='来源', comment='来源')


# 全国企业表企业税务信用
class EnterpriseTaxCredit(db.Model):
    __tablename_ = "company_tax_credit"
    __table_args__ = ({'comment': '全国企业表企业税务信用数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='全国企业表企业税务信用主键id',
                   comment='全国企业表企业税务信用主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')
    evaluation_annual = db.Column(db.String(32), doc='评价年度', comment='评价年度')
    identify_number = db.Column(db.String(32), doc='纳税人识别号', comment='纳税人识别号')
    level = db.Column(db.String(32), doc='纳税信用等级', comment='纳税信用等级')
    evaluate = db.Column(db.String(32), doc='单位评价', comment='单位评价')


# 全国企业表企业进出口信用
class EnterpriseImportExport(db.Model):
    __tablename_ = "company_import_export"
    __table_args__ = ({'comment': '全国企业表企业进出口信用数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='全国企业表企业进出口信用主键id',
                   comment='全国企业表企业进出口信用主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')
    register_customs = db.Column(db.String(32), doc='注册海关', comment='注册海关')
    business_category = db.Column(db.String(32), doc='经营类别', comment='经营类别')
    register_date = db.Column(db.DateTime, doc='注册日期', comment='注册日期')


# 全国企业表企业供应商
class EnterpriseSupplier(db.Model):
    __tablename_ = "company_supplier"
    __table_args__ = ({'comment': '全国企业表企业供应商数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='全国企业表企业供应商主键id',
                   comment='全国企业表企业供应商主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')
    supplier_name = db.Column(db.String(32), doc='客户名字', comment='客户名字')
    buy_rate = db.Column(db.String(32), doc='采购占比', comment='采购占比')
    buy_money = db.Column(db.String(32), doc='采购金额', comment='采购金额')
    open_time = db.Column(db.DateTime, doc='公开时间', comment='公开时间')
    data_source = db.Column(db.String(32), doc='数据来源', comment='数据来源')
    relation = db.Column(db.String(32), doc='关联关系', comment='关联关系')
    lng = db.Column(db.String(100), doc='经度', comment='经度')  # 经度
    lat = db.Column(db.String(100), doc='纬度', comment='纬度')  # 纬度


# 全国企业表企业客户
class EnterpriseCustomer(db.Model):
    __tablename_ = "company_customer"
    __table_args__ = ({'comment': '全国企业表企业客户数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='全国企业表企业客户主键id', comment='全国企业表企业客户主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')
    customer_name = db.Column(db.String(32), doc='客户', comment='客户')
    sales_rate = db.Column(db.String(32), doc='销售占比', comment='销售占比')
    sales_money = db.Column(db.String(32), doc='销售金额', comment='销售金额')
    open_time = db.Column(db.DateTime, comment='公开时间')
    data_source = db.Column(db.String(32), doc='数据来源', comment='数据来源')
    relation = db.Column(db.String(32), doc='关联关系', comment='关联关系')


# 全国企业表企业详情股权出质
class EnterpriseStock(db.Model):
    __tablename_ = "company_stock"
    __table_args__ = ({'comment': '全国企业表企业详情股权出质'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='股权出质主键id', comment='股权出质主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')

    num = db.Column(db.String(32), doc='等级编号', comment='等级编号')
    person = db.Column(db.String(10), doc='出质人', comment='出质人')
    from_company = db.Column(db.String(25), doc='出质股权的企业', comment='出质股权的企业')
    pledgee = db.Column(db.String(10), doc='质权人', comment='质权人')
    amount = db.Column(db.Float, doc='出质股权数额（万元）', comment='出质股权数额（万元）')
    datatime = db.Column(db.DateTime, doc='登记日期', comment='登记日期')
    status = db.Column(db.String(10), doc='状态', comment='状态')


# 全国企业表企业详情股权质押
class EnterprisePledge(db.Model):
    __tablename_ = "company_pledge"
    __table_args__ = ({'comment': '全国企业表企业详情股权质押'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='股权质押主键id', comment='股权质押主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')

    person1 = db.Column(db.String(10), doc='质押人', comment='质押人')
    jion_company = db.Column(db.String(25), doc='参股企业', comment='参股企业')
    person2 = db.Column(db.String(10), doc='质押权人', comment='质押权人')
    number = db.Column(db.Float, doc='质押股份总额（股）', comment='质押股份总额（股）')
    amount = db.Column(db.Float, doc='质押股份市值（元）', comment='质押股份市值（元）')
    status = db.Column(db.String(10), doc='状态', comment='状态')
    datatime = db.Column(db.DateTime, doc='公告日期', comment='公告日期')


# 全国企业表企业详情担保信息
class EnterpriseDanbao(db.Model):
    __tablename_ = "company_danbao"
    __table_args__ = ({'comment': '全国企业表企业详情担保信息'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='担保信息主键id', comment='担保信息主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')

    person1 = db.Column(db.String(10), doc='被担保方', comment='被担保方')
    person2 = db.Column(db.String(10), doc='担保方', comment='担保方')
    method = db.Column(db.String(10), doc='方式', comment='方式')
    amount = db.Column(db.Float, doc='担保金额（万元）', comment='担保金额（万元）')
    datatime = db.Column(db.DateTime, doc='公告日期', comment='公告日期')


# 全国企业表企业详情专利信息
class EnterprisePatent(db.Model):
    __tablename_ = "company_patent"
    __table_args__ = ({'comment': '全国企业表企业详情专利信息'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='专利信息主键id', comment='专利信息主键id')  # 企业
    company_id = db.Column(db.Integer, doc='全国企业表企业id', comment='全国企业表企业id')

    name = db.Column(db.String(50), doc='发明名称', comment='发明名称')
    type = db.Column(db.String(20), doc='类型', comment='类型')
    status = db.Column(db.String(20), doc='法律状态', comment='法律状态')
    num = db.Column(db.String(50), doc='申请号', comment='申请号')
    day = db.Column(db.DateTime, doc='申请日', comment='申请日')
    open_num = db.Column(db.String(20), doc='公开号', comment='公开号')
    datatime = db.Column(db.DateTime, doc='公开日期', comment='公开日期')
    person = db.Column(db.String(20), doc='发明人', comment='发明人')


# 产业导航目录
class Industry(db.Model):
    __tablename_ = "industry"
    __table_args__ = ({'comment': '产业导航目录表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='产业主键id', comment='产业主键id')
    name = db.Column(db.String(32), doc='导航名', comment='导航名')  # 导航名
    oname = db.Column(db.String(32), doc='导航真名', comment='导航真名')  # 导航真名
    nid = db.Column(db.Integer, doc='导航id(对应企业数据表中的f_type)', comment='导航id(对应企业数据表中的f_type)')  # 导航id(导航在企业数据中的id)
    fid = db.Column(db.Integer, doc='表中fid', comment='表中fid')  # 表中fid
    statu = db.Column(db.Integer, doc='启用状态1启用，2禁用', comment='启用状态1启用，2禁用')  # 启用状态1启用，2禁用
    info = db.Column(db.Text, doc='行业简介', comment='行业简介')  # 行业简介
    companys = db.Column(db.Integer, doc='行业下企业数', comment='行业下企业数')  # 行业下企业数
    entities = db.Column(db.Integer, doc='行业所包含细分行业实体数', comment='行业所包含细分行业实体数')  # 行业所包含细分行业实体数


# 招商驾驶舱-区县经济指标表/产业载体-行政区
class City(BaseModel, db.Model):
    __tablename__ = "city"
    __table_args__ = ({'comment': '招商驾驶舱-区县经济指标表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='经济指标主键id', comment='经济指标主键id')
    area = db.Column(db.String(255), doc='区县名称', comment='区县名称')  # 区县名称
    size = db.Column(db.Float, doc='区县面积', comment='区县面积')  # 区县面积
    year = db.Column(db.Integer, doc='年份', comment='年份')  # 年份
    people = db.Column(db.Integer, doc='人口', comment='人口')  # 人口
    GDP = db.Column(db.Float, doc='GDP(万元)', comment='GDP(万元)')  # GDP(万元)
    addscale = db.Column(db.Float, doc='规上工业增加值', comment='规上工业增加值')  # 规上工业增加值
    investment = db.Column(db.Float, doc='固定投资资产', comment='固定投资资产')  # 固定投资资产
    retail = db.Column(db.Float, doc='社会消费品零售总额', comment='社会消费品零售总额')  # 社会消费品零售总额
    in_out = db.Column(db.Float, doc='进出口总额', comment='进出口总额')  # 进出口总额
    public = db.Column(db.Float, doc='一般公共预算支出', comment='一般公共预算支出')  # 一般公共预算支出
    people_out = db.Column(db.Float, doc='居然人均可支配收入', comment='居然人均可支配收入')  # 居然人均可支配收入
    people_per = db.Column(db.Float, doc='居民消费价格指数', comment='居民消费价格指数')  # 居民消费价格指数
    public_in = db.Column(db.Float, doc='一般公共预算收入', comment='一般公共预算收入')  # 一般公共预算收入
    info = db.Column(db.Text, doc='基本信息', comment='基本信息')  # 基本信息
    question = db.Column(db.Text, doc='标注', comment='标注')  # 标注
    flag = db.Column(db.Integer, comment='数据标识 1整年数据，2阶段数据')  # 人口


# 招商驾驶舱-各区县视频
class Video(db.Model):
    __tablename__ = "video"
    __table_args__ = ({'comment': '招商驾驶舱-各区县视频表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='视频主键id', comment='视频主键id')
    district = db.Column(db.String(32), nullable=False, doc='区县名称', comment='区县名称')
    image_url = db.Column(db.String(255), doc='视频封面', comment='视频封面')
    video_id1 = db.Column(db.String(255), doc='视频链接', comment='视频链接')


# 招商驾驶舱-项目数量/金额产业分布图
class NewProject(BaseModel, db.Model):
    __tablename__ = "newproject"
    __table_args__ = ({'comment': '招商驾驶舱-项目数量/金额产业分布表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')  # id
    type = db.Column(db.String(255), doc='产业类型', comment='产业类型')  # 产业类型
    number = db.Column(db.Integer, doc='项目个数', comment='项目个数')  # 项目个数
    money = db.Column(db.Float, doc='价格', comment='价格')  # 价格
    district = db.Column(db.String(255), doc='区县名称', comment='区县名称')  #


# 招商驾驶舱-招商引资作战图
class Examine(BaseModel, db.Model):
    __tablename__ = "examine"
    __table_args__ = ({'comment': '招商驾驶舱-招商引资作战图表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')  # id
    district = db.Column(db.String(255), doc='县区', comment='县区')  # 县区
    year = db.Column(db.Integer, doc='年', comment='年')  # 年
    month = db.Column(db.Integer, doc='月', comment='月')  # 月
    project_num = db.Column(db.Integer, doc='项目个数', comment='项目个数')  # 项目个数
    sign_aim = db.Column(db.Integer, doc='签约目标', comment='签约目标')  # 签约目标

    sign_finnish = db.Column(db.Float, doc='签约的完成', comment='签约的完成')  # 签约的完成
    sign_grade = db.Column(db.Float, doc='签约的得分，签约项目投资额完成率', comment='签约的得分，签约项目投资额完成率')  # 签约的得分，签约项目投资额完成率
    start_finish = db.Column(db.Float, doc='开工实际投资', comment='开工实际投资')  # 开工实际投资
    start_grade = db.Column(db.Float, doc='开工项目完成得分，开工项目计划投资额完成率',
                            comment='开工项目完成得分，开工项目计划投资额完成率')  # 开工项目完成得分，开工项目计划投资额完成率
    invest_finish = db.Column(db.Float, doc='固投已完成', comment='固投已完成')  # 固投已完成
    invest_grade = db.Column(db.Float, doc='固投得分，开工项目到位资金额', comment='固投得分，开工项目到位资金额')  # 固投得分，开工项目到位资金额

    start_num = db.Column(db.Integer, doc='项目的开工个数', comment='项目的开工个数')  # 项目的开工个数
    start_num_grade = db.Column(db.Float, doc='签约开工率得分', comment='签约开工率得分')  # 签约开工率得分

    start_aim = db.Column(db.Float, doc='开工计划金额', comment='开工计划金额')  # 开工计划金额
    invest_aim = db.Column(db.Float, doc='固投目标金额', comment='固投目标金额')  # 固投目标金额
    invest_aim_f = db.Column(db.Float, doc='非固投目标金额', comment='非固投目标金额')  # 非固投目标金额
    invest_finish_f = db.Column(db.Integer, doc='非固投已完成', comment='非固投已完成')  # 非固投已完成
    invest_grade_f = db.Column(db.Float, doc='非固投得分', comment='非固投得分')  # 非固投得分
    grade = db.Column(db.Integer, doc='综合排名', comment='综合排名')  # 综合排名


# 招商驾驶舱-重点招商项目表
class Project(BaseModel, db.Model):
    __tablename__ = "project"
    __table_args__ = ({'comment': '招商驾驶舱-重点招商项目表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='工程id', comment='工程id')  # 工程id
    project_stage_id = db.Column(db.Integer, doc='项目阶段id：落地项目为1，招商项目为2', comment='项目阶段id：落地项目为1，招商项目为2')
    name = db.Column(db.String(255), doc='工程名', comment='工程名')  # 一.工程名
    district = db.Column(db.String(255), doc='区县名称', comment='区县名称')  # 二.县区
    type = db.Column(db.String(255), doc='工程类型', comment='工程类型')  # 三.(1)工程类型
    money = db.Column(db.String(255), doc='项目投资金额', comment='项目投资金额')  # 三.(2)项目投资金额
    background = db.Column(db.Text, doc='项目背景', comment='项目背景')  # 三.(3)项目背景
    content = db.Column(db.Text, doc='项目的具体内容', comment='项目的具体内容')  # 三.(4)项目的具体内容
    way = db.Column(db.Text, doc='拟引资方式及内容', comment='拟引资方式及内容')  # 四.拟引资方式及内容
    company = db.Column(db.String(255), doc='招商单位', comment='招商单位')  # 五.(1)招商单位
    contact = db.Column(db.String(255), doc='联系方式', comment='联系方式')  # 五.(2)联系方式
    email = db.Column(db.String(255), doc='电子邮箱', comment='电子邮箱')  # 五.(3)电子邮箱


# 招商驾驶舱-产业政策
class InduPolicy(db.Model):
    __tablename_ = "indupolicy"
    __table_args__ = ({'comment': '招商驾驶舱-产业政策'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='政策主键id', comment='政策主键id')
    name = db.Column(db.String(255), doc='政策名', comment='政策名')  # 政策名
    post_num = db.Column(db.String(255), doc='发文字号', comment='发文字号')  # 发文字号
    industry = db.Column(db.String(32), doc='行业名称', comment='行业名称')  # 行业名称
    pubdate = db.Column(db.DateTime, doc='发布时间', comment='发布时间')  # 发布时间
    year = db.Column(db.Integer, doc='发布年份', comment='发布年份')  # 发布年份
    url = db.Column(db.String(255), doc='外链接', comment='外链接')  # 外链接
    file = db.Column(db.String(255), doc='本地文件位置', comment='本地文件位置')  # 本地文件位置
    category = db.Column(db.String(32), doc='政策类型', comment='政策类型')  # 政策类型
    org = db.Column(db.String(255), doc='政策发布机构', comment='政策发布机构')  # 政策发布机构
    district = db.Column(db.String(32), doc='政策发布地区', comment='政策发布地区')  # 政策发布地区
    body = db.Column(db.Text, doc='正文', comment='正文')  # 正文（2021-4-13改为发布机构筛选条件）
    navigator = db.Column(db.String(255), doc='导航一', comment='导航一')  # 导航一
    navigat = db.Column(db.String(255), doc='导航二', comment='导航二')  # 导航二
    level = db.Column(db.String(255), doc='级别', comment='级别')  # 级别
    sorts = db.Column(db.Integer, doc='机构排序', comment='机构排序')  # 机构排序
    sorts_level = db.Column(db.Integer, doc='发布机构排序', comment='发布机构排序')  # 发布机构排序


# 产业现状图谱-园区数据表
class Induzone(db.Model):
    __tablename_ = "induzone"
    __table_args__ = ({'comment': '产业现状图谱-产业载体园区信息表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='园区主键id', comment='园区主键id')
    name = db.Column(db.String(255), doc='园区名称', comment='园区名称')  # 园区姓名
    image = db.Column(db.String(255), doc='园区图片',
                      comment='园区图片')  # 园区单图链接http://192.168.132.165:5000/static/images/garden.jpg
    phone = db.Column(db.String(255), doc='园区联系电话', comment='园区联系电话')  # 园区联系电话
    email = db.Column(db.String(30))  # 邮箱
    address = db.Column(db.String(255), doc='园区地址', comment='园区地址')  # 园区地址
    industry_position = db.Column(db.String(255), doc='产业定位', comment='产业定位')  # 产业定位
    gdp = db.Column(db.Float, doc='GDP(亿元)', comment='GDP(亿元)')  # GDP(亿元)
    represent = db.Column(db.Text, doc='代表企业', comment='代表企业')  # 代表企业
    introduct = db.Column(db.Text, doc='园区简介', comment='园区简介')  # 园区简介
    level = db.Column(db.String(32), doc='园区级别', comment='园区级别')  # 园区级别
    district = db.Column(db.String(255), doc='园区区县', comment='园区区县')  # 园区区县
    charge = db.Column(db.String(255), doc='园区负责人', comment='园区负责人')  # 园区负责人
    site = db.Column(db.String(255), doc='园区所在地', comment='园区所在地')  # 园区所在地
    industry_type = db.Column(db.String(255), doc='产业类型', comment='产业类型')  # 产业类型
    area = db.Column(db.Float, doc='占地面积(平方公里)', comment='占地面积(平方公里)')  # 占地面积(平方公里)
    # subzone = db.Column(db.String(255), doc='上级园区', comment='上级园区')  # 上级园区
    # mainzone = db.Column(db.Text, doc='主要产业园区', comment='主要产业园区')  # 主要产业园区
    # zpolicy = db.Column(db.Text, doc='产业政策', comment='产业政策')  # 产业政策
    lng = db.Column(db.String(255), doc='经度', comment='经度')  # 经度
    lat = db.Column(db.String(255), doc='纬度', comment='纬度')  # 纬度
    cluster = db.Column(db.String(255), doc='产业集群', comment='产业集群')  # 产业集群
    cate = db.Column(db.String(255), doc='园区类型', comment='园区类型')  # 园区类型

    # 园区信息
    # acreage = db.Column(db.Float, doc='规划面积（平方公里）', comment='规划面积（平方公里）')  # 规划面积（平方公里）
    # actarea = db.Column(db.Float, doc='建成面积（平方公里）', comment='建成面积（平方公里）')  # 建成面积（平方公里）
    # tax = db.Column(db.Float, doc='税收要求（万元/亩）', comment='税收要求（万元/亩）')  # 税收要求（万元/亩）
    out_power = db.Column(db.Float, doc='产出强度（万元/亩）', comment='产出强度（万元/亩）')  # 产出强度（万元/亩）
    # indu_need = db.Column(db.String(255), doc='行业要求', comment='行业要求')  # 行业要求
    # fiscal = db.Column(db.Float, doc='财政收入（亿元）', comment='财政收入（亿元）')  # 财政收入（亿元）
    invest_power = db.Column(db.Float, doc='亩均投资强度（万元/亩）', comment='亩均投资强度（万元/亩）')  # 投资强度（万元/亩）
    value_product = db.Column(db.Float, doc='亩均年产值（万元/亩）', comment='亩均年产值（万元/亩）')  # 亩均年产值（万元/亩）
    tax = db.Column(db.Float, doc='亩均年税收（万元/亩）', comment='亩均年税收（万元/亩）')  # 亩均年税收（万元/亩）
    # 投资成本
    indu_land = db.Column(db.String(255), doc='工业土地均价（万元/亩）', comment='工业土地均价（万元/亩）')  # 工业土地均价（万元/亩）
    # indu_sup = db.Column(db.String(255), doc='工业土地供应量', comment='工业土地供应量')  # 工业土地供应量
    comm_land = db.Column(db.String(255), doc='商办土地均价（万元/亩）', comment='商办土地均价（万元/亩）')  # 商办土地均价（万元/亩）
    # comm_sup = db.Column(db.String(255), doc='商办土地供应量', comment='商办土地供应量')  # 商办土地供应量

    # resident_ele_one = db.Column(db.String(255), doc='居民用电（一档）（元/度）', comment='居民用电（一档）（元/度）')  # 居民用电（一档）（元/度）
    # resident_ele_two = db.Column(db.String(255), doc='居民用电（二档）（元/度）', comment='居民用电（二档）（元/度）')  # 居民用电（二档）（元/度）
    # resident_ele_thr = db.Column(db.String(255), doc='居民用电（三档）（元/度）', comment='居民用电（三档）（元/度）')  # 居民用电（三档）（元/度）
    # comm_ele_one = db.Column(db.String(255), doc='商业用电（峰段）（元/度）', comment='商业用电（峰段）（元/度）')  # 商业用电（峰段）（元/度）
    # comm_ele_two = db.Column(db.String(255), doc='商业用电（平段）（元/度）', comment='商业用电（平段）（元/度）')  # 商业用电（平段）（元/度）
    # comm_ele_thr = db.Column(db.String(255), doc='商业用电（谷段）（元/度）', comment='商业用电（谷段）（元/度）')  # 商业用电（谷段）（元/度）
    # indu_ele_one = db.Column(db.String(255), doc='工业用电（峰段）（元/度）', comment='工业用电（峰段）（元/度）')  # 工业用电（峰段）（元/度）
    # indu_ele_two = db.Column(db.String(255), doc='工业用电（平段）（元/度）', comment='工业用电（平段）（元/度）')  # 工业用电（平段）（元/度）
    # indu_ele_thr = db.Column(db.String(255), doc='工业用电（谷段）（元/度）', comment='工业用电（谷段）（元/度）')  # 工业用电（谷段）（元/度）
    # resident_water_one = db.Column(db.String(255), doc='居民用水（一档）（元/吨）', comment='居民用水（一档）（元/吨）')  # 居民用水（一档）（元/吨）
    # resident_water_two = db.Column(db.String(255), doc='居民用水（二档）（元/吨）', comment='居民用水（二档）（元/吨）')  # 居民用水（二档）（元/吨）
    # resident_water_thr = db.Column(db.String(255), doc='居民用水（三档）（元/吨）', comment='居民用水（三档）（元/吨）')  # 居民用水（三档）（元/吨）
    # comm_water = db.Column(db.String(255), doc='商业用水（元/吨）', comment='商业用水（元/吨）')  # 商业用水（元/吨）
    # indu_water = db.Column(db.String(255), doc='工业用水（元/吨）', comment='工业用水（元/吨）')  # 工业用水（元/吨）
    # special_water = db.Column(db.String(255), doc='特种用水（元/吨）', comment='特种用水（元/吨）')  # 特种用水（元/吨）
    # resident_natgas_one = db.Column(db.String(255), doc='居民用气（一档）（元/m³）', comment='居民用气（一档）（元/m³）')  # 居民用气（一档）（元/m³）
    # resident_natgas_two = db.Column(db.String(255), doc='居民用气（二档）（元/m³）', comment='居民用气（二档）（元/m³）')  # 居民用气（二档）（元/m³）
    # resident_natgas_thr = db.Column(db.String(255), doc='居民用气（三档）（元/m³）', comment='居民用气（三档）（元/m³）')  # 居民用气（三档）（元/m³）
    # sewage = db.Column(db.String(255), doc='污水处理（元/吨）', comment='污水处理（元/吨）')  # 污水处理（元/吨）
    # wagelevel = db.Column(db.String(255), doc='最低工资标准（元/月）', comment='最低工资标准（元/月）')  # 最低工资标准（元/月）
    highmag = db.Column(db.String(255), doc='高层管理人员（元/月）', comment='高层管理人员（元/月）')  # 高层管理人员（元/月）
    middlemag = db.Column(db.String(255), doc='中级管理人员（元/月）', comment='中级管理人员（元/月）')  # 中级管理人员（元/月）
    worker = db.Column(db.String(255), doc='普通员工（元/月）', comment='普通员工（元/月）')  # 普通员工（元/月）
    # 环境配套
    # dis_freight = db.Column(db.Text, doc='距离货运站距离', comment='距离货运站距离')  # 距离货运站距离
    # dis_rail = db.Column(db.Text, doc='距离高铁站', comment='距离高铁站')  # 距离高铁站
    # dis_port = db.Column(db.Text, doc='距离港口距离', comment='距离港口距离')  # 距离港口距离
    # dis_air = db.Column(db.Text, doc='距离机场距离', comment='距离机场距离')  # 距离机场距离
    # road_trans = db.Column(db.Text, doc='公路运输', comment='公路运输')  # 公路运输
    # rail_trans = db.Column(db.Text, doc='铁路运输', comment='铁路运输')  # 铁路运输

    trans_facility = db.Column(db.Text, doc='交通配套', comment='交通配套')  # 交通配套
    goods_trans = db.Column(db.Text, doc='货物运输', comment='货物运输')  # 货物运输
    live_facility = db.Column(db.Text, doc='园区生活配套', comment='园区生活配套')  # 园区生活配套
    market = db.Column(db.Text, doc='百货商场', comment='百货商场')  # 百货商场
    hotel_bus = db.Column(db.Text, doc='酒店商务', comment='酒店商务')  # 酒店商务
    medical = db.Column(db.Text, doc='医疗机构', comment='医疗机构')  # 医疗机构
    education = db.Column(db.Text, doc='教育教育', comment='教育教育')  # 教育教育

    induenterprise = db.Column(db.Text, doc='规模以上企业', comment='规模以上企业')  # 规模以上企业
    innovate = db.Column(db.Text, doc='科研机构', comment='科研机构')  # 科研机构
    base = db.Column(db.Text, doc='双创基地', comment='双创基地')  # 双创基地
    carrier = db.Column(db.Text, doc='产业载体', comment='产业载体')  # 产业载体

    navigator = db.Column(db.String(255), doc='导航一', comment='导航一')  # 导航一
    navigat = db.Column(db.String(255), doc='导航二', comment='导航二')  # 导航二
    region = db.Column(db.String(255), doc='精确行政区', comment='精确行政区')  # 精确行政区

    # 用于园区
    gdp_point = db.Column(db.Float, doc='GDP的评分', comment='GDP的评分')  # GDP的评分
    # company_industry_point = db.Column(db.Float, doc='园区控制大小', comment='园区控制大小')  # 园区控制大小
    # warter_point = db.Column(db.Float, doc='水费的评分', comment='水费的评分')  # 水费的评分
    # electricity_point = db.Column(db.Float, doc='电费的评分', comment='电费的评分')  # 电费的评分（）
    school_point = db.Column(db.Float, doc='高校院所的评分', comment='高校院所的评分')  # 高校院所的评分
    policy_point = db.Column(db.Float, doc='政策数的评分', comment='政策数的评分')  # 政策数的评分
    live_point = db.Column(db.Float, doc='生活配套的评分', comment='生活配套的评分')  # 高校院所的评分
    traffic_point = db.Column(db.Float, doc='交通的评分', comment='交通的评分')  # 高校院所的评分
    mall_point = db.Column(db.Float, doc='购物中心的评分', comment='购物中心的评分')  # 购物中心的评分
    hotel_point = db.Column(db.Float, doc='酒店餐饮的评分', comment='酒店餐饮的评分')  # 酒店餐饮的评分
    development_zone = db.Column(db.String(255), doc='所在开发区', comment='所在开发区')  # 所在开发区
    upper_district = db.Column(db.String(255), doc='上级行政区', comment='上级行政区')  # 所在开发区
    # 辖区面积
    xiaqu_area = db.Column(db.Float, comment='辖区面积(平方公里)')
    # 人口
    people_num = db.Column(db.Float, comment='人口')
    # 载体形态
    carrier_form = db.Column(db.String(30), comment='载体形态')
    # 建筑面积
    area_structure = db.Column(db.Float, comment='辖区面积(平方公里)')


# 现状图谱-投资成本-水电气暖价格
class DistrictResource(db.Model):
    __tablename_ = "district_resource"
    __table_args__ = ({'comment': '现状图谱-投资成本-水电气暖价格'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='园区主键id', comment='园区主键id')
    district = db.Column(db.String(255), doc='区县名称', comment='区县名称')  # 园区姓名

    # 居民生活用水（元 / 立方米）
    domestic_water = db.Column(db.Float, comment='居民生活用水（元 / 立方米）')
    # 非居民用水（元 / 立方米）
    not_Domestic_water = db.Column(db.Float, comment='非居民用水（元 / 立方米）')
    # 特种用水（元 / 立方米）
    special_water = db.Column(db.Float, comment='特种用水（元 / 立方米）')
    # 电价
    # 居民生活电价
    electricity_life = db.Column(db.Float, comment='居民生活电价')
    # 商业电价
    electricity_commercial = db.Column(db.Float, comment='商业电价')
    # 工业电价
    electricity_industrial = db.Column(db.Float, comment='工业电价')
    # 燃气
    # 城市居民用气（元 / 立方米）
    gas_city = db.Column(db.Float, comment='城市居民用气（元 / 立方米）')
    # 工商业用气（元 / 立方米）
    gas_commercial = db.Column(db.Float, comment='工商业用气（元 / 立方米）')
    # 采暖成本
    # 居民用热（元 / 平方米）
    hot_life = db.Column(db.Float, comment='居民用热（元 / 平方米）')
    # 非居民用热（元 / 平方米）
    not_hot_life = db.Column(db.Float, comment='非居民用热（元 / 平方米）')


# 现状图谱-投资成本-劳动力成本
class LaborCost(db.Model):
    __tablename_ = "labor_cost"
    __table_args__ = ({'comment': '现状图谱-投资成本-劳动力成本'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='园区主键id', comment='园区主键id')
    district = db.Column(db.String(255), doc='区县名称', comment='区县名称')  # 园区姓名

    # 企业职工各种费率(人社局)
    # 养老保险
    endowment_insurance1 = db.Column(db.Float, comment='养老保险公司缴xx %')
    endowment_insurance2 = db.Column(db.Float, comment='养老保险个人缴xx %')

    # 失业保险
    unemployment_insurance1 = db.Column(db.Float, comment='失业保险公司缴XX %')
    unemployment_insurance2 = db.Column(db.Float, comment='失业保险个人缴X %')

    # 工伤保险
    injury_insurance1 = db.Column(db.Float, comment='工伤保险公司缴XX %')
    injury_insurance2 = db.Column(db.Float, comment='工伤保险个人缴X %')

    # 生育保险
    maternity_insurance1 = db.Column(db.Float, comment='生育保险公司缴XX %')
    maternity_insurance2 = db.Column(db.Float, comment='生育保险个人缴X %')

    # 医疗保险
    medical_insurance1 = db.Column(db.Float, comment='医疗保险公司缴XX %')
    medical_insurance2 = db.Column(db.Float, comment='医疗保险个人缴X %')

    # 工资标准(人社局)
    # 月最低工资工资标准
    month_wage_rates = db.Column(db.Float, comment='月最低工资工资标准')

    # 非全日制用工小时最低工资标准
    hour_wage_rates = db.Column(db.Float, comment='非全日制用工小时最低工资标准')


# 创新资源-高等院校(985, 211, 本科，高职专高)
class College(db.Model):
    __tablename_ = "college"
    __table_args__ = ({'comment': '创新资源-高等院校表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True)  # 省市区层级分类

    category = db.Column(db.String(255))  # 学校类型 211工程院校1,普通本科2,高职专高3
    cateid = db.Column(db.Integer)  # 学校类型 211工程院校1,普通本科2,高职专高3
    name = db.Column(db.String(255))  # 高等院校高职专科等机构名称
    ccode = db.Column(db.String(255))  # 院校代码
    buildate = db.Column(db.String(255))  # 创建时间
    nature = db.Column(db.String(255))  # 办学性质
    clas = db.Column(db.String(255))  # 学校类别
    feature = db.Column(db.String(255))  # 学校特色
    charge = db.Column(db.String(255))  # 主管部门
    major = db.Column(db.String(255))  # 专业设置
    faculty = db.Column(db.String(255))  # 院系设置
    introduct = db.Column(db.Text)  # 院校简介
    address = db.Column(db.String(255))  # 所在地址
    lng = db.Column(db.String(255))  # 经度
    lat = db.Column(db.String(255))  # 纬度

    navigator = db.Column(db.String(255))  # 导航一
    navigat = db.Column(db.String(255))  # 导航二

    research = db.Column(db.String(255))  # 研究方向
    admin = db.Column(db.String(255))  # 所在行政区
    developarea = db.Column(db.String(255))  # 所在开发区


# 创新资源-科研机构
class Scientific(db.Model):
    __tablename_ = "scientific"
    __table_args__ = ({'comment': '创新资源-科研机构表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    name = db.Column(db.String(255))  # 机构现名
    introduct = db.Column(db.Text)  # 机构简介
    buildate = db.Column(db.DateTime)  # 创建时间
    telephone = db.Column(db.String(255))  # 电话
    fax = db.Column(db.String(255))  # 传真
    postcode = db.Column(db.String(255))  # 邮编
    address = db.Column(db.String(255))  # 所在地址
    lng = db.Column(db.String(255))  # 经度
    lat = db.Column(db.String(255))  # 纬度

    research = db.Column(db.String(255))  # 研究方向
    admin = db.Column(db.String(255))  # 所在行政区
    developarea = db.Column(db.String(255))  # 所在开发区

    navigator = db.Column(db.String(255))  # 导航一
    navigat = db.Column(db.String(255))  # 导航二


# 创新资源-创新平台（实验室等）
class Lab(db.Model):
    __tablename_ = "lab"
    __table_args__ = ({'comment': '创新资源-创新平台表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True)

    cate = db.Column(db.String(255))  # 创新平台分类
    cateid = db.Column(db.Integer)  # 创新平台分类id
    # 工程技术中心1,国际科技合作基地2,企业技术中心3,双创基地4,院士工作站5,质量检测与测量中心6,重点实验室7，
    # 会议会展8研究院9，研发中心10，联合实验室11，检测中心12

    name = db.Column(db.String(255))  # 机构现名
    fax = db.Column(db.String(255))  # 传真
    postcode = db.Column(db.String(255))  # 邮编
    introduct = db.Column(db.Text)  # 机构简介
    address = db.Column(db.String(255))  # 所在地址
    lng = db.Column(db.String(255))  # 经度
    lat = db.Column(db.String(255))  # 纬度
    city = db.Column(db.String(255))  # 城市

    navigator = db.Column(db.String(255))  # 导航一
    navigat = db.Column(db.String(255))  # 导航二

    research = db.Column(db.String(255))  # 研究方向
    admin = db.Column(db.String(255))  # 所在行政区
    developarea = db.Column(db.String(255))  # 所在开发区


# 产业载体-土地数据
class ZaitiLand(db.Model):
    __tablename_ = "zaiti_land"
    __table_args__ = ({'comment': '产业载体-土地数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    num = db.Column(db.String(20))
    name = db.Column(db.String(20))
    addr = db.Column(db.String(50))
    nature = db.Column(db.String(20))
    acreage = db.Column(db.String(50))
    target = db.Column(db.String(255))
    age_limit = db.Column(db.String(20))
    industry_type = db.Column(db.String(255))
    telephone = db.Column(db.String(20))
    navigator = db.Column(db.String(10))
    navigat = db.Column(db.String(20))


# 产业载体-楼宇数据
class ZaitiBuild(db.Model):
    __tablename_ = "zaiti_build"
    __table_args__ = ({'comment': '产业载体-楼宇数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    name = db.Column(db.String(20))
    addr = db.Column(db.String(50))
    acreage = db.Column(db.String(50))
    buide_type = db.Column(db.String(20))
    industry_type = db.Column(db.String(255))
    rate = db.Column(db.String(20))
    telephone = db.Column(db.String(20))
    navigator = db.Column(db.String(10))
    navigat = db.Column(db.String(20))


# 产业载体-楼宇数据
class ZaitiFactory(db.Model):
    __tablename_ = "zaiti_factory"
    __table_args__ = ({'comment': '产业载体-厂房数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    name = db.Column(db.String(20))
    addr = db.Column(db.String(50))
    acreage = db.Column(db.String(50))
    structure = db.Column(db.String(20))
    height = db.Column(db.String(20))
    bearing = db.Column(db.String(50))
    new_level = db.Column(db.String(20))
    other = db.Column(db.String(255))
    industry_type = db.Column(db.String(20))
    factory_type = db.Column(db.String(20))
    telephone = db.Column(db.String(20))
    navigator = db.Column(db.String(10))
    navigat = db.Column(db.String(20))


# 雷达评估页行业在不同地区的测评分数
class EvaluationNodeData(db.Model):
    __tablename_ = "evaluation_data"
    __table_args__ = ({'comment': '产业招商雷达-产业评估'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')
    c_type = db.Column(db.Integer, index=True, doc='行业id', comment='行业id')  # 行业id

    province = db.Column(db.String(32), index=True, doc='省份', comment='省份')  # 省份
    city = db.Column(db.String(32), index=True, doc='城市', comment='城市')  # 城市
    district = db.Column(db.String(32), index=True, doc='区域', comment='区域')  # 区域

    product = db.Column(db.String(32), index=True, doc='产品名', comment='产品名')  # 产品名
    score = db.Column(db.Integer, doc='综合分', comment='综合分')  # 综合分


# 选址评分部分--企业聚合度
class CompanyIndustryPoint(db.Model):
    __tablename_ = "company_industry_point"
    __table_args__ = ({'comment': '选址评分部分--企业聚合度数据表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')
    district = db.Column(db.String(32), doc='区县园区名', comment='区县园区名')  # 区县园区名
    point = db.Column(db.Float, doc='分数', comment='分数')  # 分数
    f_type = db.Column(db.Integer, doc='类型，对应产业导航nid', comment='类型，对应产业导航nid')  # 类型，对应产业导航nid


# 360企业画像--下拉框内容数据表（企业类型，资质，上市状态，融资轮次）
class Property(db.Model):
    __tablename_ = "property"
    __table_args__ = ({'comment': '360企业画像--下拉框内容数据表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')
    name = db.Column(db.String(32), doc='字段名', comment='字段名')  # 字段名
    nid = db.Column(db.Integer, doc='字段id', comment='字段id')  # 字段id
    sid = db.Column(db.Integer, doc='字段类型id', comment='字段类型id')  # 字段类型id 1企业类型，2企业资质，3上市状态，4融资轮次
    statu = db.Column(db.Integer, doc='状态:1启动，2禁用', comment='状态:1启动，2禁用')  # 状态 1启动，2禁用


# 查找历史表
# class SearchList(BaseModel, db.Model):
#     __tablename__ = "search_list"
#     __table_args__ = ({'comment': '360企业画像--查找历史数据表'})  # 添加表注释
#
#     id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')
#     history = db.Column(db.String(128), doc='用户搜索历史', comment='用户搜索历史')  # 用户搜索历史
#     user_id = db.Column(db.Integer, db.ForeignKey("user.id"), doc='用户id', comment='用户id')
#     section = db.relationship('User', backref=db.backref('searchlist'))  # 用户-搜索
""""""


# 用户
class User(BaseModel, db.Model):
    __tablename__ = "user"
    __table_args__ = ({'comment': '用户信息表'})  # 添加表注释

    # 基本信息
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='用户编号')
    name = db.Column(db.String(32), unique=True, comment='用户名')
    password_hash = db.Column(db.String(128), comment='加密的密码')
    real_name = db.Column(db.String(32), comment='姓名')
    mobile = db.Column(db.String(11), unique=True, comment='手机号')
    flag = db.Column(db.Integer, comment='普通1，政府2')
    status = db.Column(db.Integer, comment='0禁止，1通过，2再审，3驳回')
    # 政府信息
    position = db.Column(db.String(30), comment='职务')
    belong_organization = db.Column(db.String(30), comment='单位')
    belong_department = db.Column(db.String(30), comment='部门')
    group = db.Column(db.String(30), comment='所在组')
    leader = db.Column(db.String(32), comment='领导（直属领导）')
    # 微信    # 三方登陆唯一标识
    vxopenid = db.Column(db.String(128), unique=True, comment='微信openid')
    vxunionid = db.Column(db.String(128), unique=True, comment='微信unionid')
    # 角色(职级配置)
    # role_id = db.Column(db.Integer, db.ForeignKey("role.id"), comment='角色id')
    # role = db.relationship("Role", backref=db.backref("users"))

    org_id = db.Column(db.Integer, db.ForeignKey("organization_chart.id"))
    org = db.relationship("OrganizationChart", backref=db.backref("users"))

    # 用户关注行业（图谱）
    industry = db.relationship("Industry", secondary="user_industry", backref=db.backref('bindustry'))
    # 用户关注太原企业
    company = db.relationship("Company", secondary="user_company", backref=db.backref("bcompany"))
    # 用户关注全国企业
    enterprise = db.relationship("Enterprise", secondary="user_enterprise", backref=db.backref("benterprise"))

    @property
    def password(self):
        # 设置为只写的方式
        # return self.password_hash
        raise AttributeError('这个属性只能设置，不能读取')

    @password.setter
    def password(self, value):
        self.password_hash = generate_password_hash(value)

    def check_password(self, password):
        '''检验密码的正确性，传入参数为用户登录时填写的密码'''
        return check_password_hash(self.password_hash, password)

    # 生成token
    @staticmethod
    def create_token(user_id):
        """
        生成token,生成方式（ 内部配置的私钥+有效期+用户的id ）
        :param user_id: 用户id
        :return:
        """
        # 第一个参数是内部的私钥,写在配置里,第二个参数是有效期（秒）
        s = Serializer(Config.SECRET_KEY, expires_in=Config.TOKEN_EXPIRATION)
        # 接收用户id转换与编码
        token = s.dumps({"id": user_id}).decode('ascii')
        print(token)
        return token


# 后台管理用户
class Bstage(BaseModel, db.Model):
    __tablename__ = "bstage"

    # 基本信息
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)  # 用户编号
    name = db.Column(db.String(32), unique=True)  # 用户名
    password_hash = db.Column(db.String(128))  # 加密的密码

    # 角色(职级配置)
    role = db.Column(db.Integer)  # 1超级管理员，2 系统运维员

    @property
    def password(self):
        # 设置为只写的方式
        # return self.password_hash
        raise AttributeError('这个属性只能设置，不能读取')

    @password.setter
    def password(self, value):
        self.password_hash = generate_password_hash(value)

    def check_password(self, password):
        '''检验密码的正确性，传入参数为用户登录时填写的密码'''
        return check_password_hash(self.password_hash, password)


# 角色（职位，职级）
class Role(BaseModel, db.Model):
    __tablename__ = "role"
    __table_args__ = ({'comment': '用户角色表'})  # 添加表注释

    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='企业')
    role_name = db.Column(db.String(255), unique=True, comment='角色名')
    role = db.Column(db.String(10), unique=True, comment='权限值 000000 0位位职级123，后面为权限01')
    info = db.Column(db.String(255), comment='权限说明')


# 政府机构表（自关联多对多）
# class GovOrganization(db.Model):
#     __tablename__ = "government_organization"
#     # 基本信息
#     id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')
#     pnums = db.Column(db.Integer, doc='人数', comment='人数')
#     charge = db.Column(db.String(20), doc='机构负责人', comment='机构负责人')
#     charge_phone = db.Column(db.String(20), doc='负责人电话', comment='负责人电话')
#     function = db.Column(db.Text, doc='部门职能', comment='部门职能')
#     # name = db.Column(db.String(30), nullable=False, view_index=True, doc='新增时间', comment='新增时间')
#     name = db.Column(db.String(30), doc='新增时间', comment='新增时间')
#     add_person = db.Column(db.String(20), doc='新增人', comment='新增人')
#     add_time = db.Column(db.DateTime, doc='新增时间', comment='新增时间')
#     edit_person = db.Column(db.String(20), doc='编辑人', comment='编辑人')
#     edit_time = db.Column(db.DateTime, doc='编辑时间', comment='编辑时间')
#     # followed 我的下级，管制谁，followers 我的上级，谁管制我
#     # followed = db.relationship('Government', secondary=followers, primaryjoin=(followers.c.follower_id == id),
#     #                            secondaryjoin=(followers.c.followed_id == id),
#     #                            backref=db.backref('followers', lazy='dynamic'), lazy='dynamic')
#
#
# # 政府部门（政府-部门 一对多）
# class GovDepartment(db.Model):
#     __tablename__ = "government_department"
#     # 基本信息
#     id = db.Column(db.Integer, primary_key=True, autoincrement=True, doc='主键id', comment='主键id')
#     pnums = db.Column(db.Integer, doc='人数', comment='人数')  # 人数
#     charge = db.Column(db.String(255), doc='部门负责人', comment='部门负责人')  # 部门负责人
#     charge_phone = db.Column(db.String(255), doc='负责人电话', comment='负责人电话')  # 负责人电话
#     function = db.Column(db.Text, doc='负责人电话', comment='负责人电话')  # 部门职能
#     name = db.Column(db.String(255), nullable=False, doc='政府机构部门名称', comment='政府机构部门名称')  # 政府机构部门名称
#     # gid = db.Column(db.Integer) # 政府id
#     goverment_org_id = db.Column(db.Integer, db.ForeignKey("government_organization.id"), doc='外键id，机构id',
#                                  comment='外键id，机构id')
#     goverment = db.relationship('GovOrganization', backref=db.backref('government_departments'))  # 政府1-N部门


# 政府机构之间的上下级统筹
followers = db.Table('followers',
                     db.Column('follower_id', db.Integer, db.ForeignKey('government.id'), primary_key=True),
                     db.Column('followed_id', db.Integer, db.ForeignKey('government.id'), primary_key=True)
                     )


# 政府机构表（自关联多对多）
class Government(db.Model):
    __tablename__ = "government"
    # 基本信息
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    pnums = db.Column(db.Integer)  # 人数
    charge = db.Column(db.String(255))  # 机构负责人
    charge_phone = db.Column(db.String(255))  # 负责人电话
    function = db.Column(db.Text)  # 部门职能
    name = db.Column(db.String(255), nullable=False, index=True)  # 政府机构单位名称
    # followed 我的下级，管制谁，followers 我的上级，谁管制我
    followed = db.relationship('Government', secondary=followers, primaryjoin=(followers.c.follower_id == id),
                               secondaryjoin=(followers.c.followed_id == id),
                               backref=db.backref('followers', lazy='dynamic'), lazy='dynamic')


# 政府部门（政府-部门 一对多）
class Section(db.Model):
    __tablename__ = "section"
    # 基本信息
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    pnums = db.Column(db.Integer)  # 人数
    charge = db.Column(db.String(255))  # 部门负责人
    charge_phone = db.Column(db.String(255))  # 负责人电话
    function = db.Column(db.Text)  # 部门职能
    name = db.Column(db.String(255), nullable=False)  # 政府机构部门名称
    # gid = db.Column(db.Integer) # 政府id
    goverment_id = db.Column(db.Integer, db.ForeignKey("government.id"))
    goverment = db.relationship('Government', backref=db.backref('sections'))  # 政府1-N部门


# 政府组（政府1-N部门1-N组）
class Group(db.Model):
    __tablename__ = "group"
    # 基本信息
    id = db.Column(db.Integer, primary_key=True, autoincrement=True)
    pnums = db.Column(db.Integer)  # 人数
    charge = db.Column(db.String(255))  # 组负责人
    charge_phone = db.Column(db.String(255))  # 负责人电话
    function = db.Column(db.Text)  # 部门职能
    name = db.Column(db.String(255), nullable=False)  # 政府机构部门下组的名称
    # sid = db.Column(db.Integer) # 部门id
    section_id = db.Column(db.Integer, db.ForeignKey("section.id"))
    section = db.relationship('Section', backref=db.backref('groups'))  # 部门1-N组


# 组织结构
class OrganizationChart(db.Model):
    __tablename__ = "organization_chart"
    __table_args__ = ({'comment': '后台管理系统---组织结构表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    one_org = db.Column(db.String(30), comment='一级组织机构')
    two_department = db.Column(db.String(30), comment='二级部门')
    three_role = db.Column(db.String(30), comment='三级角色')
    page_permission_id = db.Column(db.String(255), comment='页面权限id列表')
    fun_permission_id = db.Column(db.String(255), comment='功能权限id列表')


"""招商资源管理"""


# 四库管理-项目信息表
class SikuProject(db.Model):
    __tablename_ = "siku_project"
    __table_args__ = ({'comment': '四库管理-项目信息表'})  # 添加表注释
    # 项目标签
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    flag = db.Column(db.Integer, comment='1为线索库，2为对接库，3为签约库，4为开工库')
    is_delete = db.Column(db.Integer, comment='逻辑删除 1是0否')

    distribute_condition = db.Column(db.String(20), comment='分发情况')
    project_name = db.Column(db.String(200), comment='项目名称')
    investor_name = db.Column(db.String(30), comment='投资方名称')
    investor_district = db.Column(db.String(30), comment='投资方所在地')
    project_type = db.Column(db.String(30), comment='项目类型')

    industry = db.Column(db.String(20), comment='所属行业-一级产业')
    industry2 = db.Column(db.String(20), comment='所属行业-二级产业')

    provence = db.Column(db.String(30), comment='投资方省份')
    city = db.Column(db.String(30), comment='投资方市')

    investment_volume = db.Column(db.Float, comment='总投资额(万元)')
    project_year = db.Column(db.Integer, comment='项目年份')
    project_info = db.Column(db.String(300), comment='项目基本情况')
    thread_people = db.Column(db.String(30), comment='线索提供人')
    thread_people_unity = db.Column(db.String(30), comment='线索提供人单位或职务')
    thread_people_mobile = db.Column(db.String(30), comment='线索提供人联系方式')
    thread_progress = db.Column(db.String(30), comment='线索（项目）进展')

    # 线索详情
    upload_unity = db.Column(db.String(20), comment='上传部门')
    upload_people = db.Column(db.String(20), comment='上传人')
    upload_time = db.Column(db.DateTime, comment='上传时间')
    project_num = db.Column(db.String(30), comment='项目编号')
    investment_volume1 = db.Column(db.Float, comment='拟引资额(万元)')
    project_info1 = db.Column(db.String(300), comment='项目方基本情况')
    project_schedule = db.Column(db.String(20), comment='项目进展')
    stop_reason = db.Column(db.String(20), comment='暂停或终止原因')
    thread_source = db.Column(db.String(20), comment='线索来源')
    cooperation_way = db.Column(db.String(20), comment='合作方式')
    other_source = db.Column(db.String(200), comment='其他来源说明')
    remark = db.Column(db.String(200), comment='备注')

    # 对接库
    project_to_area = db.Column(db.String(20), comment='拟落地区域')
    development_area = db.Column(db.String(20), comment='开发区')
    garden = db.Column(db.String(20), comment='园区')
    project_address = db.Column(db.String(20), comment='项目详细地址')
    joint_condition = db.Column(db.String(20), comment='项目对接情况')
    project_problem = db.Column(db.String(20), comment='项目当前存在的问题')
    project_stalker = db.Column(db.String(20), comment='项目跟踪人')
    joint_person = db.Column(db.String(20), comment='对接人')
    joint_person_unity = db.Column(db.String(20), comment='对接人单位及职务')
    joint_person_mobile = db.Column(db.String(20), comment='对接人联系方式')
    project_people = db.Column(db.String(30), comment='项目方联系人')
    project_people_unity = db.Column(db.String(30), comment='项目方单位或职务')
    project_people_mobile = db.Column(db.String(30), comment='项目方联系方式')

    # 签约库
    attract_name = db.Column(db.String(30), comment='引资方名称')
    progress_condition = db.Column(db.String(20), comment='项目推进情况')
    Party_A_name = db.Column(db.String(20), comment='签约甲方名称')
    Party_A_people = db.Column(db.String(20), comment='签约甲方联系人')
    Party_A_mobile = db.Column(db.String(20), comment='签约甲方联系方式')
    Party_B_name = db.Column(db.String(20), comment='签约乙方名称')
    Party_B_people = db.Column(db.String(20), comment='签约乙方联系人')
    Party_B_mobile = db.Column(db.String(20), comment='签约乙方联系方式')
    sign_time = db.Column(db.DateTime, comment='签约时间')
    sign_year = db.Column(db.Integer, comment='签约年份')
    sign_style = db.Column(db.String(20), comment='签约方式')
    sign_explain = db.Column(db.String(255), comment='签约方式说明')

    # 开工库
    investment = db.Column(db.String(20), comment='到位资金')
    project_to_area1 = db.Column(db.String(20), comment='落地区域')
    start_time = db.Column(db.DateTime, comment='开工时间')
    start_year = db.Column(db.Integer, comment='开工年份')


# 四库管理对接库-项目进展（项目对接情况/项目推进情况）
class ProgressCondition(db.Model):
    __tablename_ = "progress_condition"
    __table_args__ = ({'comment': '四库管理对接库-项目进展（项目对接情况/项目推进情况）'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    flag = db.Column(db.Integer, comment='2为对接库，3为签约库，4为开工库')

    add_time = db.Column(db.DateTime, comment='添加时间')
    info = db.Column(db.String(20), comment='项目对接情况')
    project_id = db.Column(db.Integer, db.ForeignKey("siku_project.id"), comment='外键id，项目id')
    project = db.relationship('SikuProject', backref=db.backref('progress_conditions'))


# 四库管理对接库-相关印证资料图片/项目管理项目协议
class ProjectFile(db.Model):
    __tablename_ = "project_file"
    __table_args__ = ({'comment': '四库管理对接库-相关印证资料图片'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    flag = db.Column(db.Integer, comment='2为对接库，3为签约库，4为开工库，其他5')
    file_type = db.Column(db.Integer, comment='1文件，2图片')
    file_url = db.Column(db.String(300), comment='相关印证资料图片url')
    file_name = db.Column(db.String(20), comment='相关印证资料图片名称')
    upload_time = db.Column(db.String(20), comment='上传时间')
    upload_people = db.Column(db.String(20), comment='上传人')
    upload_people_id = db.Column(db.Integer, comment='上传人id')
    project_id = db.Column(db.Integer, db.ForeignKey("siku_project.id"), comment='外键id，项目id')
    project = db.relationship('SikuProject', backref=db.backref('project_files'))

    project_manager_id = db.Column(db.Integer, db.ForeignKey("project_management.id"), comment='外键id，项目id')
    project_manager = db.relationship('ProjectManagement', backref=db.backref('project_files1'))


# 签约库-签约的第三方
class SignThree(db.Model):
    __tablename_ = "sign_three"
    __table_args__ = ({'comment': '签约库-签约的第三方'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    # flag = db.Column(db.Integer, comment='2为对接库，3为签约库，4为开工库')
    name = db.Column(db.String(20), comment='第三方名称')
    people = db.Column(db.String(20), comment='第三方联系人')
    mobile = db.Column(db.String(20), comment='第三方联系方式')
    project_id = db.Column(db.Integer, db.ForeignKey("siku_project.id"), comment='外键id，项目id')
    project = db.relationship('SikuProject', backref=db.backref('sign_threes'))


# 四库管理-项目动态数据表
class ProjectDynamicLog(db.Model):
    __tablename_ = "project_dynamic_log"
    __table_args__ = ({'comment': '四库管理-项目动态数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    project_id = db.Column(db.Integer, comment='项目id')
    operation_people = db.Column(db.String(30), comment='操作人')
    item = db.Column(db.String(255), comment='动态')
    time = db.Column(db.String(30), comment='时间')


"""项目管理"""


# 项目管理-台账信息表
class ProjectManagement(db.Model):
    __tablename_ = "project_management"
    __table_args__ = ({'comment': '项目化管理-项目信息表'})  # 添加表注释
    # 基本信息
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    is_delete = db.Column(db.Integer, comment='逻辑删除')

    project_stalker = db.Column(db.String(30), comment='项目跟踪')
    project_stalker_explain = db.Column(db.String(30), comment='项目跟踪说明')
    project_name = db.Column(db.String(30), comment='项目名称')
    district = db.Column(db.String(20), comment='项目所在地')
    development_area = db.Column(db.String(20), comment='开发区')
    attract_name = db.Column(db.String(30), comment='引资方名称')
    investor_name = db.Column(db.String(30), comment='投资方名称')
    investor_district = db.Column(db.String(30), comment='投资方所在地')
    industry = db.Column(db.String(20), comment='所属行业-一级产业')
    industry2 = db.Column(db.String(20), comment='所属行业-二级产业')
    industry_level = db.Column(db.String(20), comment='所属项目产业级别')
    investment_volume = db.Column(db.Float, comment='总投资额(万元)')
    construction_content = db.Column(db.String(30), comment='建设内容')
    project_address = db.Column(db.String(20), comment='项目选址')
    project_progress = db.Column(db.String(30), comment='项目进展')
    project_problem = db.Column(db.String(255), comment='难点情况')
    project_year = db.Column(db.Integer, comment='项目年份')
    upload_unity = db.Column(db.String(20), comment='上传部门')
    upload_people = db.Column(db.String(20), comment='上传人')
    upload_time = db.Column(db.DateTime, comment='上传时间')
    project_num = db.Column(db.String(30), comment='项目编号')
    project_source = db.Column(db.String(20), comment='项目来源')
    project_unity = db.Column(db.String(20), comment='项目申报单位')
    is_development_project = db.Column(db.String(20), comment='是否属于开发区项目')
    sign_time = db.Column(db.String(40), default='', comment='签约时间')
    start_time = db.Column(db.String(40), default='', comment='开工时间')
    end_time = db.Column(db.String(40), default='', comment='竣工时间')
    # sign_time = db.Column(db.DateTime, default='', comment='签约时间')
    # start_time = db.Column(db.DateTime, default='', comment='开工时间')
    # end_time = db.Column(db.DateTime, default='', comment='竣工时间')
    investor_rank = db.Column(db.String(30), comment='投资方排名')
    is_transf_project = db.Column(db.String(30), comment='是否转型项目')
    country = db.Column(db.String(30), comment='投资方国别')
    provence = db.Column(db.String(30), comment='投资方省份')
    city = db.Column(db.String(30), comment='投资方市')
    job_num = db.Column(db.Integer, comment='带动就业岗位（个）')
    new_value = db.Column(db.Float, comment='新增产值（亿元）')
    revenue = db.Column(db.Float, comment='贡献税收（亿元）')

    use_land = db.Column(db.String(30), comment='项目用地情况（亩）')
    new_land = db.Column(db.String(30), comment='其中新增用地（亩）')
    construction_nature = db.Column(db.String(30), comment='建设性质（亩）')
    is_fixed_investment = db.Column(db.String(30), comment='固定资产投资项目')
    investment_year = db.Column(db.String(20), comment='资金到位本年累计（万元）')
    investment_history = db.Column(db.String(20), comment='资金到位历史累计（万元）')
    cooperation_way = db.Column(db.String(20), comment='合作方式')
    set_project_status = db.Column(db.String(20), comment='项目立项状态')
    investor_people = db.Column(db.String(30), comment='投资方联系人')
    investor_mobile = db.Column(db.String(30), comment='投资方电话')
    investor_address = db.Column(db.String(30), comment='投资方联系地址')


# 项目化管理-山西省地市指标表
class ShanxiTarget(db.Model):
    __tablename_ = "shanxi_target"
    __table_args__ = ({'comment': '项目化管理-山西省地市指标表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    is_delete = db.Column(db.Integer, comment='逻辑删除')
    data_time = db.Column(db.String(30), comment='数据年份')
    upload_time = db.Column(db.String(30), comment='上传日期')
    upload_unit = db.Column(db.String(30), comment='上传部门')
    upload_people = db.Column(db.String(30), comment='上传人')
    district_name = db.Column(db.String(30), comment='区县名称')

    # 签约项目情况
    sign_money = db.Column(db.Float, comment='签约金额')
    sign_target_money = db.Column(db.Float, comment='目标金额(年度目标)')
    sign_finish_rate = db.Column(db.Float, comment='完成率')
    # 新开工固定资产投资项目计划投资完成情况
    plan_invest_money = db.Column(db.Float, comment='计划投资额')
    plan_target_money = db.Column(db.Float, comment='目标计划投资额(年度目标)')
    plan_finish_rate = db.Column(db.Float, comment='计划投资额完成率')
    # 固定资产投资项目资金到位情况
    arrive_money = db.Column(db.Float, comment='资金到位')
    arrive_target_money = db.Column(db.Float, comment='目标资金到位(年度目标)')
    arrive_finish_rate = db.Column(db.Float, comment='计划投资额完成率')


# 项目化管理-晋城市县（市、区）、开发区指标表
class JcTarget(db.Model):
    __tablename_ = "jc_target"
    __table_args__ = ({'comment': '项目化管理-晋城市县（市、区）、开发区指标表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    file_name = db.Column(db.String(30), comment='文件名称')
    is_delete = db.Column(db.Integer, comment='逻辑删除')
    year = db.Column(db.String(30), comment='数据年份')
    upload_time = db.Column(db.String(30), comment='上传日期')
    upload_unit = db.Column(db.String(30), comment='上传部门')
    upload_people = db.Column(db.String(30), comment='上传人')

    grade_sign = db.Column(db.Float, comment='签约项目情况赋分15')
    grade_start = db.Column(db.Float, comment='项目开工率赋分5')
    grade_plan_invest = db.Column(db.Float, comment='新开工固定资产投资项目计划投资赋分25')
    grade_arrive_target1 = db.Column(db.Float, comment='固定资产投资项目资金到位赋分25')
    grade_arrive_target0 = db.Column(db.Float, comment='非固定资产投资项目资金到位赋分5')

    district_name = db.Column(db.String(30), comment='区县名称')
    money_sign = db.Column(db.Float, comment='签约项目金额(年度目标)')
    rate_start = db.Column(db.Float, comment='项目开工率(年度目标)')
    money_plan_invest = db.Column(db.Float, comment='新开工固定资产投资项目计划投资额(年度目标)')
    money_arrive_target1 = db.Column(db.Float, comment='固定资产投资项目资金到位额(年度目标)')
    money_arrive_target0 = db.Column(db.Float, comment='非固定资产投资项目资金到位额(年度目标)')


# 项目化管理-固定资产投资项目资金到位表
class MoneyArrive(db.Model):
    __tablename_ = "money_arrive"
    __table_args__ = ({'comment': '项目化管理-固定资产投资项目资金到位表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    is_delete = db.Column(db.Integer, comment='逻辑删除')
    file_name = db.Column(db.String(30), comment='文件名称')
    data_time = db.Column(db.String(30), comment='数据年份')
    upload_time = db.Column(db.String(30), comment='上传日期')
    upload_unit = db.Column(db.String(30), comment='上传部门')
    upload_people = db.Column(db.String(30), comment='上传人')
    district_name = db.Column(db.String(30), comment='区县名称')
    arrive_money = db.Column(db.Float, comment='固定资产投资项目资金到位额')


"""以下为202212月改版后需求"""


# 招商图谱-产业链收藏
# class IndustryCollect(db.Model):
#     __tablename_ = "industry_collect"
#     __table_args__ = ({'comment': '招商图谱-产业链收藏'})  # 添加表注释
#     id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
#     industry_id = db.Column(db.Integer, comment='产业id')
#     user_id = db.Column(db.Integer, comment='用户id')
#     collect_time = db.Column(db.String(30), comment='收藏时间')
#     industry_name = db.Column(db.String(30), comment='收藏的产业名称')
""""""


# 用户关注行业，多-多
class UserIndustry(BaseModel, db.Model):
    __tablename__ = "user_industry"

    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), primary_key=True)
    industry_id = db.Column(db.Integer, db.ForeignKey("industry.id"), primary_key=True)


# #用户关注公司，多-多
class UserCompany(BaseModel, db.Model):
    __tablename__ = "user_company"

    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), primary_key=True)
    company_id = db.Column(db.Integer, db.ForeignKey("company.id"), primary_key=True)


# #用户关注公司，多-多
class UserEnterprise(BaseModel, db.Model):
    __tablename__ = "user_enterprise"

    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), primary_key=True)
    enterprise_id = db.Column(db.Integer, db.ForeignKey("enterprise.id"), primary_key=True)


# 载体资源库-数据填报-楼宇详情表
class CarrierBuild(db.Model):
    __tablename_ = "carrier_build"
    __table_args__ = ({'comment': '载体资源库-楼宇详情表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')

    district_name = db.Column(db.String(50), comment='区域名称')
    build_name = db.Column(db.String(50), comment='楼宇名称')
    industry_name = db.Column(db.String(255), comment='产业名称')
    industry_type = db.Column(db.String(20), comment='产业类型')
    total_area = db.Column(db.String(20), comment='总面积(㎡)')
    # total_area_id = db.Column(db.Integer,
    #                           comment='总面积范围id(1:0-500㎡；2:500-1000㎡；3:1000-2000㎡；4:2000-5000㎡；5:5000-10000㎡)')
    cover_land_area = db.Column(db.String(20), comment='占地面积(㎡)')
    # cover_land_area_id = db.Column(db.Integer,
    #                                comment='占地面积范围id(1:0-500㎡；2:500-1000㎡；3:1000-2000㎡；4:2000-5000㎡；5:5000-10000㎡)')
    build_type = db.Column(db.String(20), comment='楼宇类型')
    # build_type_id = db.Column(db.Integer, comment='楼宇类型id(1:商铺；2:写字楼)')
    rent_rate = db.Column(db.String(20), comment='出租率')
    # rent_rate_id = db.Column(db.Integer, comment='出租率id(1:0-50％；2:50-80％；3:80-90％；4:90-100％；)')
    layer_num = db.Column(db.Integer, comment='总层数（层，填写整数）')
    detail_address = db.Column(db.String(50), comment='详细地址')
    lng = db.Column(db.String(50), comment='经度')
    lat = db.Column(db.String(50), comment='维度')

    construction_time = db.Column(db.String(20), default='', comment='建设时间(例如：2022年)')
    upload_time = db.Column(db.DateTime, default='', comment='数据上传时间')
    # 权限相关
    upload_people = db.Column(db.String(20), comment='数据上传人')
    upload_people_id = db.Column(db.String(20), comment='数据上传人用户id')
    upload_people_role_id = db.Column(db.String(20), comment='数据上传人权限id')
    upload_people_belong = db.Column(db.String(100), comment='数据上传人所属区县、机构、部门')

    fix_time = db.Column(db.DateTime, default='', comment='数据修改时间')
    fix_people = db.Column(db.String(20), comment='数据修改人')
    build_status = db.Column(db.String(20), comment='楼宇状态')
    # rent_money = db.Column(db.String(20), comment='租金（万元）')
    rent_money = db.Column(db.String(20), comment='租金范围（元/平米/天）')
    wuye_money = db.Column(db.String(20), comment='物业费（元/平米/天）')
    audit_status = db.Column(db.Integer, comment='审核状态 0为未审核；1为审核通过；2为提交；3为驳回；4未通过')
    audit_message = db.Column(db.String(50), comment='审核附言')

    build_info = db.Column(db.String(255), comment='楼宇介绍')
    inside_picture_url = db.Column(db.String(255), comment='内部照片url')
    outer_picture_url = db.Column(db.String(255), comment='外部照片url')
    price_url = db.Column(db.String(255), comment='平面图url')

    car_space_num = db.Column(db.Integer, comment='车位数（个，填写整数）')
    lift_num = db.Column(db.Integer, comment='电梯数（部，填写整数）')
    rentout_status = db.Column(db.String(20), comment='出租状态')
    linkman = db.Column(db.String(20), comment='联系人')
    link_mobile = db.Column(db.String(20), comment='联系方式')
    build_area = db.Column(db.String(20), comment='建筑面积(㎡)')

    empty_area = db.Column(db.String(20), comment='闲置面积(㎡)')
    build_structure = db.Column(db.String(20), comment='建筑结构')
    owner = db.Column(db.String(20), comment='权属人')
    attract_status = db.Column(db.String(20), comment='拟招引状态')
    attract_industry_status = db.Column(db.String(20), comment='招引业态')
    cooperation_model = db.Column(db.String(20), comment='合作模式')
    attract_advantage = db.Column(db.String(255), comment='招商优势')
    policy = db.Column(db.String(255), comment='优惠政策')

    # settled_company = db.Column(db.String(255), comment='入住企业名称（多个，列表形式，顿号分隔）')
    # surround_facility = db.Column(db.String(255), comment='周边配套（多个，列表形式，顿号分隔）')
    # education_name = db.Column(db.String(255), comment='教育机构名称（多个，列表形式，顿号分隔）')
    # medical_name = db.Column(db.String(255), comment='医疗设施名称（多个，列表形式，顿号分隔）')
    # park_name = db.Column(db.String(255), comment='大型公园名称（多个，列表形式，顿号分隔）')
    # shangfu_center = db.Column(db.String(255), comment='商服中心（多个，列表形式，顿号分隔）')
    # administration_center = db.Column(db.String(255), comment='行政中心（多个，列表形式，顿号分隔）')
    # transportation = db.Column(db.String(255), comment='交通设施（多个，列表形式，顿号分隔）')


# 载体资源库-数据填报-土地详情表
class CarrierLand(db.Model):
    __tablename_ = "carrier_land"
    __table_args__ = ({'comment': '载体资源库-土地详情表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    district_name = db.Column(db.String(50), comment='区域名称')
    land_name = db.Column(db.String(50), comment='土地名称')
    industry_name = db.Column(db.String(255), comment='产业名称')
    land_nature = db.Column(db.String(20), comment='土地性质)')
    land_nature_id = db.Column(db.Integer, comment='土地性质id(1:农用地；2:商业用地；3:建设用地；4:旅游用地；5:居民用地。)')
    detail_address = db.Column(db.String(20), comment='详细地址')
    lng = db.Column(db.String(50), comment='经度')
    lat = db.Column(db.String(50), comment='维度')
    total_area = db.Column(db.String(20), comment='总面积(㎡)')
    total_area_id = db.Column(db.Integer,
                              comment='总面积范围id(1:0-500㎡；2:500-1000㎡；3:1000-2000㎡；4:2000-5000㎡；5:5000-10000㎡)')
    industry_type = db.Column(db.String(20), comment='产业类型')
    jing_area = db.Column(db.String(20), comment='净面积(㎡)')
    transfer_year = db.Column(db.String(20), comment='出让年限（年）')
    transfer_year_id = db.Column(db.Integer, comment='出让年限范围id(1:0-5年；2:5-10年；3:10-15年；4:15-20年；5:20年以上)')

    land_code = db.Column(db.String(50), comment='土地编码')
    upload_time = db.Column(db.DateTime, default='', comment='数据上传时间')
    upload_people = db.Column(db.String(20), comment='数据上传人名字')
    upload_people_id = db.Column(db.String(20), comment='数据上传人用户id')
    upload_people_role_id = db.Column(db.String(20), comment='数据上传人权限id')
    upload_people_belong = db.Column(db.String(100), comment='数据上传人所属区县、机构、部门')
    fix_time = db.Column(db.DateTime, default='', comment='数据修改时间')
    fix_people = db.Column(db.String(20), comment='数据修改人名字')
    land_status = db.Column(db.Integer, comment='土地状态 0为闲置；1为占用')
    audit_status = db.Column(db.Integer, comment='审核状态 0为未审核；1为审核通过；2为提交；3为驳回；4未通过')
    audit_message = db.Column(db.String(50), comment='审核附言')
    attract_status = db.Column(db.String(20), comment='拟招引状态')

    linkman = db.Column(db.String(20), comment='联系人')
    link_mobile = db.Column(db.String(20), comment='联系方式')
    land_info = db.Column(db.String(255), comment='土地介绍')
    inside_picture_url = db.Column(db.String(255), comment='内部照片url')
    outer_picture_url = db.Column(db.String(255), comment='外部照片url')
    plot_ratio = db.Column(db.String(20), comment='容积率')
    sales_price = db.Column(db.String(20), comment='销售均价（毛坯）（万元）')
    max_sales_price = db.Column(db.String(20), comment='最高销售单价（万元）')
    max_car_space_price = db.Column(db.String(20), comment='车位最高销售单价')
    plan_target = db.Column(db.String(20), comment='规划指标')
    plan_condition = db.Column(db.String(20), comment='规划条件')
    price_url = db.Column(db.String(255), comment='规划图url')

    surround_facility = db.Column(db.String(255), comment='周边配套（多个，列表形式，顿号分隔）')
    education_name = db.Column(db.String(255), comment='教育机构名称（多个，列表形式，顿号分隔）')
    medical_name = db.Column(db.String(255), comment='医疗设施名称（多个，列表形式，顿号分隔）')
    park_name = db.Column(db.String(255), comment='大型公园名称（多个，列表形式，顿号分隔）')
    shangfu_center = db.Column(db.String(255), comment='商服中心（多个，列表形式，顿号分隔）')
    administration_center = db.Column(db.String(255), comment='行政中心（多个，列表形式，顿号分隔）')
    policy = db.Column(db.String(255), comment='相关政策（多个，列表形式，顿号分隔）')


# 载体资源库-数据填报-厂房详情表
class CarrierFactory(db.Model):
    __tablename_ = "carrier_factory"
    __table_args__ = ({'comment': '载体资源库-厂房详情表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    district_name = db.Column(db.String(50), comment='区域名称')
    factory_name = db.Column(db.String(50), comment='厂房名称')
    # is_standard = db.Column(db.Integer, comment='是否为标准化厂房(是1否0)')
    # rent_status = db.Column(db.Integer, comment='是否对外租赁(是1否0)')
    is_standard = db.Column(db.String(10), comment='是否为标准化厂房(是或否)')
    rent_status = db.Column(db.String(10), comment='是否对外租赁(是或否)')
    # rent_money = db.Column(db.String(20), comment='租金（万元）')
    rent_money = db.Column(db.String(20), comment='租金范围（元/平米/天）')
    detail_address = db.Column(db.String(20), comment='详细地址')
    factory_area = db.Column(db.String(20), comment='建筑面积(㎡)')
    cover_land_area = db.Column(db.String(20), comment='占地面积(㎡)')
    cover_land_area_id = db.Column(db.Integer,
                                   comment='占地面积范围id(1:0-500㎡；2:500-1000㎡；3:1000-2000㎡；4:2000-5000㎡；5:5000-10000㎡)')
    industry_name = db.Column(db.String(255), comment='产业名称')
    property_type = db.Column(db.String(50), comment='产权类型')

    construction_time = db.Column(db.String(20), default='', comment='建设时间(例如：2022年)')
    upload_time = db.Column(db.DateTime, default='', comment='数据上传时间')
    upload_people = db.Column(db.String(20), comment='数据上传人名字')
    upload_people_id = db.Column(db.String(20), comment='数据上传人用户id')
    upload_people_role_id = db.Column(db.String(20), comment='数据上传人权限id')
    upload_people_belong = db.Column(db.String(100), comment='数据上传人所属区县、机构、部门')

    fix_time = db.Column(db.DateTime, default='', comment='数据修改时间')
    fix_people = db.Column(db.String(20), comment='数据修改人名字')
    factory_status = db.Column(db.String(50), comment='厂房状态')
    audit_status = db.Column(db.String(20), comment='审核状态 0为未审核；1为审核通过；2为提交；3为驳回；4未通过；')
    audit_message = db.Column(db.String(50), comment='审核附言')
    lng = db.Column(db.String(50), comment='经度')
    lat = db.Column(db.String(50), comment='维度')
    factory_info = db.Column(db.String(255), comment='厂房介绍')
    inside_picture_url = db.Column(db.String(255), comment='内部照片url')
    outer_picture_url = db.Column(db.String(255), comment='外部照片url')
    price_url = db.Column(db.String(255), comment='平面图url')

    # owner = db.Column(db.String(20), comment='权属人')
    empty_area = db.Column(db.String(20), comment='闲置面积(㎡)')
    factory_structure = db.Column(db.String(20), comment='建筑结构')
    width = db.Column(db.String(20), comment='跨度（米）')
    high = db.Column(db.String(20), comment='层高（米）')
    bearing = db.Column(db.String(20), comment='承重（500kg/m2）')
    linkman = db.Column(db.String(20), comment='联系人')
    link_mobile = db.Column(db.String(20), comment='联系方式')
    # is_electric = db.Column(db.Integer, comment='是否已通电(是1否0,填写整数)')
    # is_water = db.Column(db.Integer, comment='是否已通水(是1否0,填写整数)')
    # is_warm = db.Column(db.Integer, comment='是否已通暖(是1否0,填写整数)')
    # is_gas = db.Column(db.Integer, comment='是否已通燃气(是1否0,填写整数)')
    # is_network = db.Column(db.Integer, comment='是否已通网络(是1否0,填写整数)')
    # is_lift = db.Column(db.Integer, comment='是否有电梯(是1否0)')
    is_electric = db.Column(db.String(10), comment='是否已通电(是或否)')
    is_water = db.Column(db.String(10), comment='是否已通水(是或否)')
    is_warm = db.Column(db.String(10), comment='是否已通暖(是或否)')
    is_gas = db.Column(db.String(10), comment='是否已通燃气(是或否)')
    is_network = db.Column(db.String(10), comment='是否已通网络(是或否)')
    is_lift = db.Column(db.String(10), comment='是否有电梯(是或否)')
    lift_num = db.Column(db.Integer, comment='电梯数量（部,填写整数）')
    is_car_space = db.Column(db.Integer, comment='是否有车位(是1否0)')
    # car_space_num = db.Column(db.Integer, comment='车位数（个,填写整数）')
    layer_num = db.Column(db.Integer, comment='总层数（层,填写整数）')
    attract_status = db.Column(db.String(20), comment='拟招引状态')
    cooperation_model = db.Column(db.String(20), comment='合作模式')
    attract_advantage = db.Column(db.String(255), comment='招商优势')
    policy = db.Column(db.String(255), comment='相关政策')
    # settled_company = db.Column(db.String(255), comment='入住企业名称（多个，列表形式，顿号分隔）')
    # surround_facility = db.Column(db.String(255), comment='周边配套（多个，列表形式，顿号分隔）')
    # education_name = db.Column(db.String(255), comment='教育机构名称（多个，列表形式，顿号分隔）')
    # medical_name = db.Column(db.String(255), comment='医疗设施名称（多个，列表形式，顿号分隔）')
    # park_name = db.Column(db.String(255), comment='大型公园名称（多个，列表形式，顿号分隔）')
    # shangfu_center = db.Column(db.String(255), comment='商服中心（多个，列表形式，顿号分隔）')
    # administration_center = db.Column(db.String(255), comment='行政中心（多个，列表形式，顿号分隔）')
    # transportation = db.Column(db.String(255), comment='交通设施（多个，列表形式，顿号分隔）')


# 载体资源库-产业地图-各载体类型面积数据表
# class CarrierArea(db.Model):
#     __tablename_ = "carrier_area"
#     __table_args__ = ({'comment': '载体资源库-产业地图-各载体类型面积数据表'})  # 添加表注释
#     id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
#     district_name = db.Column(db.String(20), comment='区域名称')
#     lng = db.Column(db.String(50), comment='经度')
#     lat = db.Column(db.String(50), comment='维度')
#     build_area = db.Column(db.String(20), comment='楼宇当前总面积(㎡)')
#     build_current_empty_area = db.Column(db.String(20), comment='楼宇当前闲置面积(㎡)')
#     build_last_empty_area = db.Column(db.String(20), comment='楼宇去年闲置面积(㎡)')
#     factory_area = db.Column(db.String(20), comment='厂房当前总面积(㎡)')
#     factory_current_empty_area = db.Column(db.String(20), comment='厂房当前闲置面积(㎡)')
#     factory_last_empty_area = db.Column(db.String(20), comment='厂房去年闲置面积(㎡)')
#     plan_land_store_area = db.Column(db.String(20), comment='计划土地储备面积(亩)')
#     plan_land_supply_area = db.Column(db.String(20), comment='计划土地供应面积(亩)')
#     plan_land_stock_area = db.Column(db.String(20), comment='计划土地存量面积(亩)')
#     land_area = db.Column(db.String(20), comment='土地当前总面积(亩)')
#     land_current_empty_area = db.Column(db.String(20), comment='土地当前闲置面积(亩)')
#     land_last_empty_area = db.Column(db.String(20), comment='土地去年闲置面积(亩)')
""""""


# 载体资源库-产业地图-水电气热能源数据表
class CarrierEnergy(db.Model):
    __tablename_ = "carrier_energy"
    __table_args__ = ({'comment': '载体资源库-产业地图-水电气热能源数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    district_name = db.Column(db.String(20), comment='区域名称')
    electric_one = db.Column(db.String(50), comment='居民用电（一档）(元/度)')
    electric_two = db.Column(db.String(50), comment='居民用电（二档）(元/度)')
    electric_three = db.Column(db.String(20), comment='居民用电（三档）(元/度)')
    business_electric_feng = db.Column(db.String(20), comment='商业用电（峰段）(元/度)')
    business_electric_ping = db.Column(db.String(20), comment='商业用电（平段）(元/度)')
    business_electric_gu = db.Column(db.String(20), comment='商业用电（谷段）(元/度)')
    industrial_electric_feng = db.Column(db.String(20), comment='工业电价（峰段）(元/度)')
    industrial_electric_ping = db.Column(db.String(20), comment='工业电价（平段）(元/度)')
    industrial_electric_gu = db.Column(db.String(20), comment='工业电价（谷段）(元/度)')
    life_water_one = db.Column(db.String(20), comment='居民用水（一档）(元/吨)')
    life_water_two = db.Column(db.String(20), comment='居民用水（二档）(元/吨)')
    life_water_three = db.Column(db.String(20), comment='居民用水（三档）(元/吨)')
    business_water = db.Column(db.String(20), comment='商业用水(元/吨)')
    industrial_water = db.Column(db.String(20), comment='工业用水(元/吨)')
    special_water = db.Column(db.String(20), comment='特种用水(元/吨)')
    life_gas_one = db.Column(db.String(20), comment='居民用气（一档）(元/m³)')
    life_gas_two = db.Column(db.String(20), comment='居民用气（二档）(元/m³)')
    life_gas_three = db.Column(db.String(20), comment='居民用气（三档）(元/m³)')


# 载体资源库-产业地图-各区县闲置环比按月份区分显示图数据表
# class CarrierAreaData(db.Model):
#     __tablename_ = "carrier_rate_data"
#     __table_args__ = ({'comment': '载体资源库-产业地图-各区县各载体闲置面积数据表'})  # 添加表注释
#     id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
#     district_name = db.Column(db.String(20), comment='区域名称')
#     land_year = db.Column(db.String(20), comment='土地全年闲置面积')
#     land_one = db.Column(db.String(20), comment='土地第一季度闲置面积')
#     land_two = db.Column(db.String(20), comment='土地第二季度闲置面积')
#     land_three = db.Column(db.String(20), comment='土地第三季度闲置面积')
#     land_four = db.Column(db.String(20), comment='土地第四季度闲置面积')
#
#     factory_year = db.Column(db.String(20), comment='厂房全年闲置面积')
#     factory_one = db.Column(db.String(20), comment='厂房第一季度闲置面积')
#     factory_two = db.Column(db.String(20), comment='厂房第二季度闲置面积')
#     factory_three = db.Column(db.String(20), comment='厂房第三季度闲置面积')
#     factory_four = db.Column(db.String(20), comment='厂房第四季度闲置面积')
#
#     build_year = db.Column(db.String(20), comment='楼宇全年闲置面积')
#     build_one = db.Column(db.String(20), comment='楼宇第一季度闲置面积')
#     build_two = db.Column(db.String(20), comment='楼宇第二季度闲置面积')
#     build_three = db.Column(db.String(20), comment='楼宇第三季度闲置面积')
#     build_four = db.Column(db.String(20), comment='楼宇第四季度闲置面积')


# 记录操作日志得数据表
class OperationLog(db.Model):
    __tablename_ = "operation_log"
    __table_args__ = ({'comment': '载体资源库-记录操作日志的数据表'})  # 添加表注释
    id = db.Column(db.Integer, primary_key=True, autoincrement=True, comment='主键id')
    operation_time = db.Column(db.DateTime, default='', comment='操作时间')
    operation_people = db.Column(db.String(20), comment='操作人名字')
    operation_people_id = db.Column(db.String(20), comment='操作人id')
    operation_people_belong = db.Column(db.String(100), comment='操作人所属区县、机构、部门')
    operation_mobile = db.Column(db.String(20), comment='操作人电话')
    operation_message = db.Column(db.String(255), comment='操作信息')
    section = db.Column(db.String(20), comment='操作人职务')
    remark = db.Column(db.String(20), comment='备注：是或否（是否是每月二十五号前进行提报导入）')
    action = db.Column(db.String(20), comment='记录"导入"的动作用于备注（每月十五号前是否进行提报导入）的是和否')
    read_type = db.Column(db.Integer, comment='消息类型（用于消息提醒） 0未读，1已读，3已读状态一天后转为历史消息')
    time = db.Column(db.DateTime, default='', comment='转为已读状态的时间（用于消息提醒）')
