import os
import redis
import logging
from flask import Flask
from config import config_map
from flask_sqlalchemy import SQLAlchemy
from flask_session import Session
from flask_wtf import CSRFProtect
from flask_cors import CORS
from flask_mail import Mail
from logging.handlers import RotatingFileHandler
from apps.utils.commons import RegexConverter
# from apps.util import RegexConverter
from flask_qiniustorage import Qiniu

db = SQLAlchemy()  # 创建数据库连接
qiniu_store = Qiniu()  # 创建七牛云连接
redis_store = None  # 创建变量避免导入错误，在函数中进行初始化

# 配置日志信息
# 创建日志记录器，指明日志保存的路径、每个日志文件的最大大小、保存的日志文件个数上限
file_log_handler = RotatingFileHandler("logs/logs", maxBytes=1024 * 1024 * 100, backupCount=10)
stream_log_handler = logging.StreamHandler()
# 创建日志记录的格式       日志等级    输入日志信息的文件名 行数    日志信息
formatter = logging.Formatter('%(levelname)s %(filename)s:%(lineno)d %(message)s')
# 为刚创建的日志记录器设置日志记录格式
file_log_handler.setFormatter(formatter)
# 为全局的日志工具对象（flask app使用的）添加日志记录器
logging.getLogger().addHandler(file_log_handler)
# 为全局的日志工具对象（flask app使用的）添加控制台显示记录器
logging.getLogger().addHandler(stream_log_handler)
# 设置日志的记录等级
# logging.basicConfig(level=logging.DEBUG)  # 调试debug级，会受flask的debug=True影响，强制忽略设置的等级
logging.basicConfig(level=logging.INFO)  # 调试debug级，会受flask的debug=True影响，强制忽略设置的等级


# 跨域支持
def after_request(resp):
    resp.headers['Access-Control-Allow-Origin'] = '*'
    resp.headers['Access-Control-Allow-Credentials'] = 'TRUE'
    resp.headers[
        'Access-Control-Allow-Headers'] = 'x-requested-with,content-type,token'  # 允许的请求header
    resp.headers['Access-Control-Allow-Methods'] = 'GET,POST,OPTIONS'
    return resp


# 允许的请求header示例
# 'Content-Type,Authorization,X-Requested-With,token,application/octet-stream'
#  x-requested-with,content-type
# "Accept,Authorization,Cache-Control,Content-Type,DNT,If-Modified-Since,Keep-Alive,Origin,User-Agent,X-Requested-With"
def creat_app(config_name):
    '''
    :param config_name: str 配置模式的名称('develop', 'product')
    :return: object flask的app对象
    '''
    app = Flask(__name__)
    app.after_request(after_request)

    config_class = config_map.get(config_name)
    app.config.from_object(config_class)

    # 绑定SQLAlchemy的app对象
    db.init_app(app)

    # 七牛
    qiniu_store.init_app(app)

    # 初始化redis工具，创建Redis连接对象用于缓存
    global redis_store
    redis_store = redis.StrictRedis(host=config_class.REDIS_HOST, port=config_class.REDIS_PORT,
                                    db=2, password=config_class.REDIS_PASS)

    # cors跨域插件,是否允许发送cookies
    # CORS(app, supports_credentials=True)
    # CORS(app, resources={r"/.*": {"origins": ["http://bigxigua.net","http://localhost:8006"]}})
    CORS(app, resources=r'/*')

    # 邮箱
    Mail(app)

    # 利用flask_session扩展将session数据保存到redis中
    Session(app)

    # 为flask补充CSRF防护
    # CSRFProtect(app)

    # 为flask添加自定义的转换器
    app.url_map.converters['re'] = RegexConverter

    # 注册蓝图，推迟导入，防止循环嵌套
    from apps.view_index import api_attract  # 招商驾驶舱
    from apps.view_user import api_user
    from apps.view_atlas import api_atlas
    from apps.view_radar import api_radar
    from apps.view_map import api_map
    from apps.view_360company import api_portraits
    from apps.view_choose import api_address
    from apps.manage import api_power
    from apps.inves_manage import api_manage
    # from apps.view_mobile import api_mobile

    app.register_blueprint(api_user, url_prefix='/api/user')
    app.register_blueprint(api_radar, url_prefix='/api/radar')  # 产业招商雷达
    app.register_blueprint(api_atlas, url_prefix='/api/atlas')  # 产业现状图谱
    app.register_blueprint(api_map, url_prefix='/api/map')  # 产业招商地图

    app.register_blueprint(api_attract, url_prefix='/api/attract/industry')  # 招商驾驶舱
    app.register_blueprint(api_portraits, url_prefix="/api/360company")  # 360企业画像
    app.register_blueprint(api_address, url_prefix="/api/address")  # 选址评估
    app.register_blueprint(api_power, url_prefix="/api/power")
    app.register_blueprint(api_manage, url_prefix="/api/manage")
    # app.register_blueprint(api_mobile, url_prefix="/api/mobile")

    return app
